/**
 * VGC Dojo Companion - Popup Script
 * Opens side panel and displays stats
 */

class PopupController {
  constructor() {
    this.stats = {
      battleCount: 0,
      turnsAnalyzed: 0
    };
  }

  async init() {
    await this.loadStats();
    this.bindEvents();
    this.updateUI();
  }

  /**
   * Load stats from Chrome storage
   */
  async loadStats() {
    try {
      const result = await chrome.storage.local.get('stats');
      if (result.stats) {
        this.stats = { ...this.stats, ...result.stats };
      }
    } catch (error) {
      console.log('Using default stats');
    }
  }

  /**
   * Bind event listeners
   */
  bindEvents() {
    const openPanelBtn = document.getElementById('openPanelBtn');
    if (openPanelBtn) {
      openPanelBtn.addEventListener('click', async () => {
        // Get the current tab
        const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
        
        if (tab?.url?.includes('play.pokemonshowdown.com')) {
          // Open side panel for this tab
          chrome.sidePanel.open({ tabId: tab.id });
          window.close(); // Close popup
        } else {
          // Show message that user needs to be on Showdown
          openPanelBtn.textContent = 'Go to Showdown first!';
          openPanelBtn.style.background = '#FF5B6A';
          setTimeout(() => {
            openPanelBtn.textContent = 'Open Side Panel';
            openPanelBtn.style.background = '';
          }, 2000);
        }
      });
    }
  }

  /**
   * Update UI elements with current values
   */
  updateUI() {
    document.getElementById('battleCount').textContent = this.stats.battleCount.toLocaleString();
    document.getElementById('turnsAnalyzed').textContent = this.stats.turnsAnalyzed.toLocaleString();
  }
}

// Initialize when DOM is ready
document.addEventListener('DOMContentLoaded', () => {
  const controller = new PopupController();
  controller.init();
});
