/**
 * Battle Coach - Side Panel Script
 * Receives battle state from content script and displays scored moves
 */

// Import scoring engine inline (will be present in side panel context)
// Type chart for effectiveness calculation
// Scoring constants and logic now shared from engine.js

class SidePanelController {

  constructor() {
    this.port = null;
    this.connected = false;
    this.currentState = null;
    this.isDemoMode = false;
    this.liveState = null;
    this.moveHistoryLedger = {}; // { species: Set of moves }
    this.pollingInterval = null;
  }

  static SCENARIO_PRESETS = {
    endgame: {
      turn: 12,
      format: 'DOUBLES',
      player: {
        active: [
          { species: 'Calyrex-Shadow', hpPercent: 85, usedMoves: ['Astral Barrage', 'Psychic', 'Protect', 'Nasty Plot'], availableMoves: [{name: 'Astral Barrage'}, {name: 'Psychic'}, {name: 'Protect'}, {name: 'Nasty Plot'}] }
        ],
        bench: [],
        sideConditions: {}
      },
      opponent: {
        active: [
          { species: 'Incineroar', hpPercent: 45, usedMoves: ['Fake Out', 'Flare Blitz'] },
          { species: 'Flutter Mane', hpPercent: 15, usedMoves: ['Moonblast', 'Shadow Ball'] }
        ],
        bench: [],
        sideConditions: {}
      },
      field: { weather: null, terrain: 'psychic', trickRoom: false, tailwind: true }
    },
    matchup: {
      turn: 3,
      format: 'DOUBLES',
      player: {
        active: [
          { species: 'Kyogre', hpPercent: 100, usedMoves: ['Water Spout', 'Origin Pulse', 'Ice Beam'], availableMoves: [{name: 'Water Spout'}, {name: 'Origin Pulse'}, {name: 'Ice Beam'}] }
        ],
        bench: [],
        sideConditions: {}
      },
      opponent: {
        active: [
          { species: 'Groudon', hpPercent: 100, usedMoves: ['Precipice Blades'] },
          { species: 'Incineroar', hpPercent: 100, usedMoves: ['Fake Out'] }
        ],
        bench: [],
        sideConditions: {}
      },
      field: { weather: 'rain', terrain: null, trickRoom: false }
    },
    bulk: {
      turn: 5,
      format: 'DOUBLES',
      player: {
        active: [
          { species: 'Flutter Mane', hpPercent: 100, usedMoves: ['Moonblast', 'Shadow Ball'], availableMoves: [{name: 'Moonblast'}, {name: 'Shadow Ball'}] }
        ],
        bench: [],
        sideConditions: {}
      },
      opponent: {
        active: [
          { species: 'Regice', hpPercent: 100, usedMoves: ['Ice Beam'] },
          { species: 'Probopass', hpPercent: 100, usedMoves: ['Flash Cannon'] }
        ],
        bench: [],
        sideConditions: {}
      },
      field: { weather: 'snow', terrain: null, trickRoom: true }
    },
    threat: {
      turn: 1,
      format: 'SINGLES',
      player: {
        active: [
          { species: 'Oricorio-Pau', hpPercent: 100, usedMoves: ['Roost', 'Protect'], availableMoves: [{name: 'Roost'}, {name: 'Protect'}] }
        ],
        bench: [
          { species: 'Ampharos', hpPercent: 100 },
          { species: 'Lokix', hpPercent: 100 }
        ],
        sideConditions: {}
      },
      opponent: {
        active: [
          { species: 'Pikachu', hpPercent: 100, usedMoves: ['Thunderbolt'] }
        ],
        bench: [],
        sideConditions: {}
      },
      field: { weather: null, terrain: null, trickRoom: false }
    }
  };

  init() {
    console.log('🥋 Battle Coach initializing...');

    // Load databases for new BattleCoach engine
    Promise.all([
      fetch(chrome.runtime.getURL('data/pokemon-stats.json')).then(r => r.json()),
      fetch(chrome.runtime.getURL('data/smogon-stats.json')).then(r => r.json())
    ]).then(([pokemonStats, smogonStats]) => {
      // Initialize BattleCoach engine
      if (typeof BattleCoach !== 'undefined') {
        BattleCoach.initialize({ pokemonStats, smogonStats });
        console.log('🥋 BattleCoach engine initialized');
      } else {
        console.error('🥋 BattleCoach not loaded!');
      }
      
      console.log('🥋 Databases loaded:', Object.keys(pokemonStats).length, 'pokemon');
    }).catch(e => console.warn('🥋 Failed to load databases:', e));

    // Load Smogon stats database for UI display
    if (typeof SmogonDB !== 'undefined') {
      SmogonDB.load().then(() => {
        console.log('🥋 SmogonDB loaded for UI');
      }).catch(e => console.warn('🥋 Failed to load SmogonDB:', e));
    }
    
    this.updateConnectionStatus('Connecting...');
    this.connectToBackground();
    this.initInfoModal();
    
    // Refresh Button
    document.getElementById('refreshBtn')?.addEventListener('click', () => {
      this.forceRefresh();
    });

    this.generateStarField();
    this.startPolling();
  }

  forceRefresh() {
    console.log('🥋 Force refreshing battle state...');
    this.updateConnectionStatus('Refreshing...');
    
    // Clear local caches and exit demo mode
    this.moveHistoryLedger = {};
    this.currentState = null;
    this.liveState = null;
    this.isDemoMode = false;
    
    // Visual feedback on refresh button
    const refreshBtn = document.getElementById('refreshBtn');
    if (refreshBtn) refreshBtn.classList.add('refreshing');
    
    // First try: Query for Showdown tabs specifically
    chrome.tabs.query({ url: 'https://play.pokemonshowdown.com/*' }, (showdownTabs) => {
      if (chrome.runtime.lastError) {
        console.log('🥋 Tab query error:', chrome.runtime.lastError.message);
      }
      
      // If we found Showdown tabs, use the first one
      const targetTabs = showdownTabs && showdownTabs.length > 0 ? showdownTabs : [];
      
      // Fallback: Also check active tab
      chrome.tabs.query({ active: true, currentWindow: true }, (activeTabs) => {
        // Combine and deduplicate
        const allTabs = [...targetTabs];
        if (activeTabs && activeTabs[0] && !allTabs.find(t => t.id === activeTabs[0].id)) {
          allTabs.push(activeTabs[0]);
        }
        
        if (allTabs.length === 0) {
          console.log('🥋 No tabs found');
          this.updateConnectionStatus('No Tab Found');
          return;
        }
        
        // Try each tab until one responds
        let responded = false;
        let tried = 0;
        
        for (const tab of allTabs) {
          chrome.tabs.sendMessage(tab.id, { type: 'forceRefresh' }, (response) => {
            tried++;
            
            if (responded) return; // Already got a good response
            
            if (chrome.runtime.lastError) {
              console.log(`🥋 Tab ${tab.id} error:`, chrome.runtime.lastError.message);
              if (tried === allTabs.length && !responded) {
                this.updateConnectionStatus('No Extension Found');
                if (refreshBtn) refreshBtn.classList.remove('refreshing');
              }
              return;
            }
            
            if (response?.state) {
              responded = true;
              console.log('🥋 Got state from tab', tab.id);
              this.handleStateUpdate(response.state);
              this.updateConnectionStatus('⚔️ BATTLE ACTIVE');
              if (refreshBtn) refreshBtn.classList.remove('refreshing');
            } else if (tried === allTabs.length && !responded) {
              // Show waiting state
              if (refreshBtn) refreshBtn.classList.remove('refreshing');
              const waitingState = document.getElementById('waitingState');
              const battleState = document.getElementById('battleState');
              waitingState?.classList.remove('hidden');
              battleState?.classList.add('hidden');
              this.updateConnectionStatus('No Battle Found');
            }
          });
        }
      });
    });
  }

  // Poll for state every 5 seconds to catch up if behind
  startPolling() {
    if (this.pollingInterval) clearInterval(this.pollingInterval);
    
    this.pollingInterval = setInterval(() => {
      // Only poll if not in an active battle state
      if (!this.currentState || !this.currentState.player) {
        console.log('🥋 Polling for battle state...');
        chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
          if (tabs[0]?.id) {
            chrome.tabs.sendMessage(tabs[0].id, { type: 'requestState' }, (response) => {
              if (response?.state) {
                this.handleStateUpdate(response.state);
              }
            });
          }
        });
      }
    }, 5000);
  }


  generateStarField() {
    const container = document.getElementById('background-layer');
    if (!container) return;

    const starsContainer = document.createElement('div');
    starsContainer.className = 'stars-container';
    starsContainer.style.position = 'absolute';
    starsContainer.style.inset = '0';
    starsContainer.style.pointerEvents = 'none';
    starsContainer.style.zIndex = '0';
    
    // Use individual elements for guaranteed visibility
    for (let i = 0; i < 60; i++) {
      const star = document.createElement('div');
      const x = Math.random() * 100;
      const y = Math.random() * 70; // Top 70%
      const size = Math.random() > 0.9 ? 2 : 1;
      const opacity = Math.random() * 0.5 + 0.3;
      const color = Math.random() > 0.8 ? '#00D9C4' : '#FFFFFF';
      
      star.style.position = 'absolute';
      star.style.left = `${x}%`;
      star.style.top = `${y}%`;
      star.style.width = `${size}px`;
      star.style.height = `${size}px`;
      star.style.backgroundColor = color;
      star.style.borderRadius = '50%';
      star.style.opacity = opacity;
      star.style.boxShadow = `0 0 ${size * 2}px ${color}`;
      
      // Subtle twinkling animation
      star.style.animation = `pulse ${2 + Math.random() * 4}s infinite ease-in-out`;
      star.style.animationDelay = `${Math.random() * 5}s`;
      
      starsContainer.appendChild(star);
    }
    
    // Mask to fade out towards floor
    starsContainer.style.webkitMaskImage = 'linear-gradient(to bottom, rgba(0,0,0,1) 0%, rgba(0,0,0,1) 40%, transparent 80%)';
    starsContainer.style.maskImage = 'linear-gradient(to bottom, rgba(0,0,0,1) 0%, rgba(0,0,0,1) 40%, transparent 80%)';
    
    container.appendChild(starsContainer);
  }

  initInfoModal() {
    const helpBtn = document.getElementById('helpBtn');
    const infoModal = document.getElementById('infoModal');
    const closeBtn = document.getElementById('closeInfoBtn');
    const overlay = infoModal?.querySelector('.modal-overlay');

    if (helpBtn && infoModal) {
      helpBtn.addEventListener('click', () => {
        infoModal.classList.remove('hidden');
      });
    }

    if (closeBtn && infoModal) {
      closeBtn.addEventListener('click', () => {
        infoModal.classList.add('hidden');
      });
    }

    if (overlay && infoModal) {
      overlay.addEventListener('click', () => {
        infoModal.classList.add('hidden');
      });
    }
  }

  loadScenario(key) {
    const scenario = SidePanelController.SCENARIO_PRESETS[key];
    if (scenario) {
      console.log(`🥋 Loading demo scenario: ${key}`);
      this.isDemoMode = true;
      this.handleStateUpdate(scenario);
      this.updateConnectionStatus(`SIM: ${key.toUpperCase()}`);
      
      const turnInfo = document.getElementById('turnInfo');
      if (turnInfo) {
        turnInfo.classList.add('demo-active');
        turnInfo.textContent = `🚨 SIMULATION MODE: ${key.toUpperCase()} 🚨`;
      }
    }
  }

  resetToLive() {
    console.log('🥋 Resetting to live state');
    this.isDemoMode = false;
    const turnInfo = document.getElementById('turnInfo');
    if (turnInfo) turnInfo.classList.remove('demo-active');
    
    if (this.liveState) {
      this.handleStateUpdate(this.liveState);
    } else {
      this.handleStateUpdate(null);
    }
  }

  connectToBackground() {
    try {
      this.port = chrome.runtime.connect({ name: 'sidepanel' });
      
      this.port.onMessage.addListener((message) => {
        if (message.type === 'stateUpdate') {
          this.handleStateUpdate(message.state);
        }
      });

      // Request initial state immediately to avoid "Connecting..." hang
      setTimeout(() => {
        chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
          if (tabs[0]?.id) {
            chrome.tabs.sendMessage(tabs[0].id, { type: 'requestState' }, (response) => {
              if (response?.state) {
                this.handleStateUpdate(response.state);
              }
            });
          }
        });
      }, 500);

      this.port.onDisconnect.addListener(() => {
        this.connected = false;
        this.updateConnectionStatus('Disconnected');
        setTimeout(() => this.connectToBackground(), 2000);
      });

      this.connected = true;
      this.updateConnectionStatus('Ready');
      
      chrome.runtime.sendMessage({ type: 'getState' }, (response) => {
        if (response?.state) {
          this.handleStateUpdate(response.state);
        }
      });
      
    } catch (e) {
      console.error('🥋 Failed to connect:', e);
      this.updateConnectionStatus('Connection failed');
    }
  }

  updateConnectionStatus(status) {
    const indicator = document.getElementById('connectionStatus');
    if (!indicator) return;
    
    indicator.className = 'status-indicator';
    
    if (status === 'Ready') {
      indicator.textContent = '✓ Ready';
      indicator.classList.add('connected');
    } else if (status.includes('Turn') || status.includes('⚔️')) {
      indicator.textContent = status;
      indicator.classList.add('battle');
    } else {
      indicator.textContent = `⏳ ${status}`;
    }
  }

  handleStateUpdate(state) {
    const waitingState = document.getElementById('waitingState');
    const battleState = document.getElementById('battleState');
    
    console.log('🥋 State update received:', state ? 'Valid' : 'Null');
    if (!state || !state.player) {
      if (!this.isDemoMode) {
        console.log('🥋 No active battle state found, showing waiting screen');
        waitingState?.classList.remove('hidden');
        battleState?.classList.add('hidden');
        this.updateConnectionStatus('Ready');
      }
      return;
    }
    
    console.log('🥋 Processing active battle state for', state.player.active.length, 'player mon(s)');
    
    if (!this.isDemoMode) {
      this.liveState = state;
    }
    
    this.currentState = state;
    
    // Update move cache from current state
    this.updateMoveCache(state);
    
    waitingState?.classList.add('hidden');
    battleState?.classList.remove('hidden');
    
    this.updateConnectionStatus(`⚔️ BATTLE ACTIVE`);
    this.renderBattleState(state);
  }

  updateMoveCache(state) {
    try {
      const processTeam = (team) => {
        if (!team?.active) return;
        team.active.forEach(p => {
          if (!p?.species) return;
          
          if (!this.moveHistoryLedger[p.species]) this.moveHistoryLedger[p.species] = new Set();
          
          // Add moves from buttons (player)
          if (p.availableMoves) {
            p.availableMoves.forEach(m => {
              if (m?.name) this.moveHistoryLedger[p.species].add(m.name);
            });
          }
          
          // Add moves from log (used)
          if (p.usedMoves) {
            p.usedMoves.forEach(m => {
              if (m) this.moveHistoryLedger[p.species].add(m);
            });
          }
        });
      };
      
      processTeam(state.player);
      processTeam(state.opponent);
    } catch (e) {
      console.error('🥋 Error updating move cache:', e);
    }
  }

  renderBattleState(state) {
    try {
      const threatBanner = document.getElementById('threatBanner');
      
      // 1. Convert state format for engine
      const engineState = this.convertStateForEngine(state);
      
      // 2. Use BattleCoach engine
      let topActions = [];
      let isThreatened = false;
      
      if (typeof BattleCoach !== 'undefined' && BattleCoach.isReady()) {
        const analysis = BattleCoach.analyze(engineState);
        console.log('🥋 Analysis result:', analysis);
        topActions = this.convertRecommendationsToActions(analysis.recommendations);
        isThreatened = (state.player?.active || []).some(p => p && p.hpPercent < 30);
      } else {
        console.warn('🥋 BattleCoach not ready!');
      }
      
      // 3. Handle Threat Banner
      if (threatBanner) {
        if (isThreatened) {
          threatBanner.classList.remove('hidden');
        } else {
          threatBanner.classList.add('hidden');
        }
      }

      // 4. Render Top Strategies
      const strategiesContainer = document.getElementById('topStrategies');
      if (strategiesContainer) {
        strategiesContainer.innerHTML = this.renderTopActions(topActions);
      }
      
      // 4. Get Speed Tiers for all active Pokemon
      let speedTiers = {};
      if (typeof SpeedPredictor !== 'undefined') {
        speedTiers = SpeedPredictor.getAllSpeedTiers(
          state.player?.active || [],
          state.opponent?.active || [],
          state.field || {}
        );
      }
      
      // 5. Render Team Details with Speed Tiers
      const opponentContainer = document.getElementById('opponentPokemon');
      if (opponentContainer && state.opponent?.active) {
        // Reverse opponent list to match mirrored visual layout (Slot 1 on right)
        const reversedOpponents = [...state.opponent.active].filter(p => p && p.species).reverse();
        const cards = reversedOpponents.map(pokemon => {
          const tierInfo = speedTiers[pokemon.species] || null;
          return this.renderPokemonCard(pokemon, true, tierInfo);
        }).join('');
        const gridClass = (reversedOpponents.length === 1) ? 'single' : '';
        opponentContainer.innerHTML = `<div class="pokemon-cards ${gridClass}">${cards}</div>`;
      }

      const playerContainer = document.getElementById('playerPokemon');
      if (playerContainer && state.player?.active) {
        const filteredPlayers = state.player.active.filter(p => p && p.species);
        const cards = filteredPlayers.map(pokemon => {
          const tierInfo = speedTiers[pokemon.species] || null;
          return this.renderPokemonCard(pokemon, false, tierInfo);
        }).join('');
        const gridClass = filteredPlayers.length === 1 ? 'single' : '';
        playerContainer.innerHTML = `<div class="pokemon-cards ${gridClass}">${cards}</div>`;
      }

      // 4. Render Field Conditions
      this.renderFieldConditions(state);
    } catch (e) {
      console.error('🥋 Error rendering battle state:', e);
    }
  }

  renderTopActions(actions) {
    if (!actions || actions.length === 0) {
      return '<div class="no-moves">No viable strategies found</div>';
    }

    return actions.map(action => {
      if (action.type === 'move') {
        return this.renderMoveItem(action);
      } else if (action.type === 'tera') {
        return this.renderTeraItem(action);
      } else {
        return this.renderUnifiedSwitchItem(action);
      }
    }).join('');
  }

  renderUnifiedSwitchItem(s) {
    const stars = s.score >= 180 ? '★★★' :
                  s.score >= 120 ? '★★☆' :
                  s.score >= 60 ? '★☆☆' : '☆☆☆';
    
    let rankClass = 'rank-neutral';
    if (s.score >= 180) rankClass = 'rank-best';
    else if (s.score >= 120) rankClass = 'rank-good';
    else if (s.score >= 60) rankClass = 'rank-neutral';
    else if (s.score >= 30) rankClass = 'rank-poor';
    else rankClass = 'rank-worst';

    return `
      <div class="move-item ${rankClass}">
        <div class="move-header">
          <span class="move-stars">${stars}</span>
          <span class="move-name">Switch to ${s.species} <span class="switch-badge">SWITCH</span></span>
        </div>
      </div>
    `;
  }

  renderTeraItem(t) {
    const stars = t.score >= 180 ? '★★★' :
                  t.score >= 120 ? '★★☆' :
                  t.score >= 60 ? '★☆☆' : '☆☆☆';
    
    let rankClass = 'rank-neutral';
    if (t.score >= 180) rankClass = 'rank-best';
    else if (t.score >= 120) rankClass = 'rank-good';
    else if (t.score >= 60) rankClass = 'rank-neutral';
    else if (t.score >= 30) rankClass = 'rank-poor';
    else rankClass = 'rank-worst';

    return `
      <div class="move-item ${rankClass} type-${t.label.replace('Tera ', '').toLowerCase()}">
        <div class="move-header">
          <span class="move-stars">${stars}</span>
          <span class="move-name">${t.label} <span class="tera-badge-action">✨ TERA</span></span>
        </div>
      </div>
    `;
  }



  /**
   * Render a single move item with type-colored background
   */
  renderMoveItem(moveScore) {
    const moveData = moveScore.moveData || MOVE_DATA[moveScore.label] || { type: 'normal', power: 0, category: 'status' };
    const typeClass = `type-${moveData.type}`;
    
    // Show category and base power
    let categoryBadge = '';
    let powerDisplay = '';
    
    if (moveData.category === 'status') {
      categoryBadge = '<span class="move-category status">SUPPORT</span>';
    } else {
      const cat = moveData.category === 'physical' ? 'PHY' : 'SPC';
      categoryBadge = `<span class="move-category ${moveData.category}">${cat}</span>`;
      powerDisplay = `<span class="move-power">${moveData.power} BP</span>`;
    }
    
    const stars = moveScore.score >= 180 ? '★★★' :
                  moveScore.score >= 120 ? '★★☆' :
                  moveScore.score >= 60 ? '★☆☆' : '☆☆☆';
    
    // Assign rank class based on score
    let rankClass = 'rank-neutral';
    if (moveScore.score >= 180) rankClass = 'rank-best';
    else if (moveScore.score >= 120) rankClass = 'rank-good';
    else if (moveScore.score >= 60) rankClass = 'rank-neutral';
    else if (moveScore.score >= 30) rankClass = 'rank-poor';
    else rankClass = 'rank-worst';
    
    // Check for STAB
    let stabBadge = '';
    if (moveData && moveData.type && moveData.category !== 'status') {
      // Check if attacker has STAB for this move type
      const attackerTypes = moveScore.attackerTypes || [];
      if (attackerTypes.includes(moveData.type)) {
        stabBadge = '<span class="stab-badge">STAB</span>';
      }
    }
    
    const targetHtml = moveScore.targetDisplay ? ` → ${moveScore.targetDisplay}` : '';
    
    return `
      <div class="move-item ${typeClass} ${rankClass}">
        <div class="move-header">
          <span class="move-stars">${stars}</span>
          <span class="move-name">${moveScore.label}${targetHtml}</span>
          ${stabBadge}
          ${categoryBadge}
          ${powerDisplay}
        </div>
      </div>
    `;
  }


  renderPokemonCard(pokemon, isOpponent = false, tierInfo = null) {
    // Guard clause for invalid pokemon data
    if (!pokemon || !pokemon.species) {
      console.warn('🥋 renderPokemonCard called with invalid pokemon:', pokemon);
      return '';
    }
    
    const hpClass = pokemon.hpPercent > 70 ? 'hp-high' : 
                    pokemon.hpPercent > 30 ? 'hp-med' : 'hp-low';
    
    // 1. Nickname vs Species
    const isNickname = pokemon.name && pokemon.name !== pokemon.species;
    const displayName = isNickname ? pokemon.name : pokemon.species;
    const subName = isNickname ? pokemon.species : '';
    
    // Speed tier badge (S/A/B/C/D/F)
    let speedBadgeHtml = '';
    if (tierInfo && tierInfo.tier) {
      const tierClass = `speed-tier-${tierInfo.tier.toLowerCase()}`;
      const trClass = tierInfo.trickRoom ? 'trick-room-active' : '';
      const priorityMarker = tierInfo.hasPriority ? '+' : '';
      speedBadgeHtml = `<span class="speed-badge ${tierClass} ${trClass}" title="Speed Tier ${tierInfo.tier}">${tierInfo.tier}${priorityMarker}</span>`;
    }

    // 2. Types - use DataAdapter
    let typesHtml = '';
    let types = null;
    
    // Use DataAdapter
    if (typeof DataAdapter !== 'undefined' && DataAdapter.isInitialized()) {
      types = DataAdapter.getTypes(pokemon.species);
    }
    
    if (types && types.length > 0) {
      typesHtml = types.map(t => `<span class="type-badge ${t.toLowerCase()}">${t.toUpperCase()}</span>`).join('');
    }

    // 3. Status
    const statusBadge = pokemon.status ? 
      `<span class="stat-badge status-badge ${pokemon.status}">${this.getStatusLabel(pokemon.status)}</span>` : '';
    
    // 4. HP
    const hpBadge = `<span class="stat-badge hp-badge ${hpClass}">${pokemon.hpPercent}% HP</span>`;

    // 5. Predicted Ability & Item (Opponents only)
    let predictedHtml = '';
    if (isOpponent && typeof SmogonDB !== 'undefined') {
      const topAbility = SmogonDB.getTopAbility(pokemon.species);
      const topItem = SmogonDB.getTopItem(pokemon.species);
      
      // Use display fields if state-reader populated them, otherwise fallback to SmogonDB + formatName
      // Use display fields if state-reader populated them, or fall back to SmogonDB
      const abilityValue = pokemon.revealedAbility || topAbility?.name || 'Unknown Ability';
      const itemValue = pokemon.revealedItem || topItem?.name || 'Unknown Item';
      
      // Smart Ability Display
      let abilityContent = '';
      let abilityTitle = '';
      const isAbilityConfirmed = !!pokemon.revealedAbility;

      if (isAbilityConfirmed) {
          abilityContent = typeof SmogonDB !== 'undefined' ? SmogonDB.formatName(abilityValue) : abilityValue;
          abilityTitle = 'Confirmed Ability';
      } else if (pokemon.possibleAbilities && pokemon.possibleAbilities.length > 0) {
          // Rank predictions
          const ranked = pokemon.possibleAbilities.map(name => {
              const data = SmogonDB.getPokemonData(pokemon.species);
              const normalized = name.toLowerCase().replace(/[^a-z0-9]/g, '');
              let usage = 0;
              if (data && data.abilities) {
                 for (const [key, val] of Object.entries(data.abilities)) {
                     if (key.toLowerCase().replace(/[^a-z0-9]/g, '') === normalized) usage = val;
                 }
              }
              return { name, usage };
          }).sort((a, b) => b.usage - a.usage);
          
          abilityContent = (typeof SmogonDB !== 'undefined' ? SmogonDB.formatName(ranked[0].name) : ranked[0].name) + '?';
          abilityTitle = 'Possible: ' + ranked.map(r => `${r.name} (${Math.round(r.usage * 100)}%)`).join(', ');
      } else {
          abilityContent = (typeof SmogonDB !== 'undefined' ? SmogonDB.formatName(abilityValue) : abilityValue) + '?';
          abilityTitle = 'Predicted Ability';
      }

      // Smart Item Display
      let itemContent = '';
      let itemTitle = '';
      const isItemConfirmed = !!pokemon.revealedItem;
      
      if (isItemConfirmed) {
          itemContent = typeof SmogonDB !== 'undefined' ? SmogonDB.formatName(itemValue) : itemValue;
          itemTitle = 'Confirmed Item';
      } else if (pokemon.possibleItems && pokemon.possibleItems.length > 0) {
          const ranked = pokemon.possibleItems.map(name => {
              const data = SmogonDB.getPokemonData(pokemon.species);
              const normalized = name.toLowerCase().replace(/[^a-z0-9]/g, '');
              let usage = 0;
              if (data && data.items) {
                 for (const [key, val] of Object.entries(data.items)) {
                     if (key.toLowerCase().replace(/[^a-z0-9]/g, '') === normalized) usage = val;
                 }
              }
              return { name, usage };
          }).sort((a, b) => b.usage - a.usage);

          itemContent = (typeof SmogonDB !== 'undefined' ? SmogonDB.formatName(ranked[0].name) : ranked[0].name) + '?';
          itemTitle = 'Possible: ' + ranked.map(r => `${r.name} (${Math.round(r.usage * 100)}%)`).join(', ');
      } else {
          itemContent = (typeof SmogonDB !== 'undefined' ? SmogonDB.formatName(itemValue) : itemValue) + '?';
          itemTitle = 'Predicted Item';
      }

      predictedHtml = `
        <div class="predicted-container">
          <span class="predicted-badge ability ${isAbilityConfirmed ? 'confirmed' : ''}" title="${abilityTitle}">
            ${abilityContent}
          </span>
          <span class="predicted-badge item ${isItemConfirmed ? 'confirmed' : ''}" title="${itemTitle}">
            ${itemContent}
          </span>
        </div>
      `;
    }

    // 6. Used Moves (Opponent only)
    let usedMovesHtml = '';
    if (isOpponent && pokemon.usedMoves && pokemon.usedMoves.length > 0) {
      usedMovesHtml = `
        <div class="used-moves-container">
          ${pokemon.usedMoves.map(m => `<span class="used-move-badge">${m}</span>`).join('')}
        </div>
      `;
    }

    return `
      <div class="pokemon-card ${isOpponent ? 'opponent' : ''}">
        <div class="pokemon-header">
          <div class="name-container">
            ${speedBadgeHtml}
            <div class="pokemon-name">${displayName}</div>
            ${subName ? `<div class="pokemon-subname">${subName}</div>` : ''}
          </div>
          <div class="types-container">
            ${typesHtml}
          </div>
          <div class="status-container">
            ${statusBadge}
            ${hpBadge}
          </div>
          ${predictedHtml}
          ${usedMovesHtml || ''}
        </div>
      </div>
    `;
  }

  getSpeciesGrades(species) {
    // Use the grades.js file (SPECIES_GRADES)
    if (typeof window.SPECIES_GRADES !== 'undefined') {
      const normalized = species.toLowerCase().replace(/[^a-z]/g, '');
      const grades = window.SPECIES_GRADES[normalized];
      if (grades) {
        return {
          power: grades.power || '?',
          bulk: grades.bulk || '?',
          support: grades.support || '?'
        };
      }
    }
    return null;
  }

  /**
   * Score a move against a target
   */
  scoreMoveAgainstTarget(move, attacker, defender) {
    // Use new engine for scoring
    return { score: 50, reasoning: "Use BattleCoach.analyze for scoring" };
  }

  scoreAllMoves(attacker, opponents) {
    // Use new engine for scoring
    return [];
  }

  getTypeEffectiveness(attackType, defenderTypes) {
    // Use new TypeChart
    if (typeof TypeChart !== 'undefined') {
      return TypeChart.getEffectivenessVsDual(attackType, defenderTypes);
    }
    return 1;
  }

  getPokemonStats(species) {
    // Use new DataAdapter
    if (typeof DataAdapter !== 'undefined' && DataAdapter.isInitialized()) {
      return DataAdapter.getBaseStats(species);
    }
    return null;
  }

  /**
   * Convert old state format to new BattleCoach engine format
   */
  convertStateForEngine(state) {
    if (!state) return null;
    
    const convertPokemon = (p, includeAvailable = false) => {
      if (!p) return null;
      const converted = {
        name: p.species,
        hp: p.hpPercent || 100,
        status: p.status || null,
        ability: p.revealedAbility || null,
        item: p.revealedItem || null,
        moves: (p.usedMoves || []),
        types: p.types || null,
        teraType: p.teraType || null,
        teraActive: p.teraActive || false,
        justSwitchedIn: p.justSwitchedIn,
        lastMove: p.lastMove
      };
      
      if (includeAvailable && p.availableMoves) {
        converted.availableMoves = p.availableMoves.map(m => m.name || m);
      }
      
      return converted;
    };
    
    return {
      turn: state.turn || 1,
      playerActive: (state.player?.active || []).map(p => convertPokemon(p, true)),
      opponentActive: (state.opponent?.active || []).map(p => convertPokemon(p, false)),
      playerBench: (state.player?.bench || []).map(p => convertPokemon(p, false)),
      opponentBench: (state.opponent?.bench || []).map(p => convertPokemon(p, false)),
      field: {
        weather: state.field?.weather || null,
        terrain: state.field?.terrain || null,
        trickRoom: state.field?.trickRoom || false,
        trickRoomTurns: state.field?.trickRoomTurns || 0,
        tailwind: state.field?.tailwind || false,
        tailwindTurns: state.field?.tailwindTurns || 0
      },
      canTera: state.player?.canTera !== false
    };
  }

  /**
   * Convert new engine recommendations to old action format for UI
   */
  convertRecommendationsToActions(recommendations) {
    if (!recommendations || recommendations.length === 0) return [];
    
    const actions = [];
    
    for (const slotRec of recommendations) {
      for (const action of (slotRec.actions || [])) {
        if (action.type === 'move') {
          const moveData = window.MoveData?.getMove(action.move) || { type: 'normal', power: 0, category: 'status' };
          actions.push({
            type: 'move',
            label: (moveData.name || action.move) + (action.isSpread ? ' (Spread)' : ''),
            score: Math.round(action.score * 100),
            reasoning: (action.rationale || []).join('. ') || '',
            targetDisplay: (action.target && action.target !== 'self') ? action.target : null,
            moveData: { type: moveData.type || 'normal', power: moveData.basePower || 0, category: moveData.category || 'status' },
            attackerTypes: [],
            risks: action.risks || []
          });
        } else if (action.type === 'switch') {
          actions.push({
            type: 'switch',
            species: action.switchTo,
            score: Math.round(action.score * 100),
            reasoning: (action.rationale || []).join('. ') || 'Better matchup'
          });
        } else if (action.type === 'tera') {
          actions.push({
            type: 'tera',
            label: `Tera ${action.teraType}`,
            score: Math.round(action.score * 100),
            reasoning: (action.rationale || []).join('. ') || 'Boost'
          });
        }
      }
    }
    
    actions.sort((a, b) => b.score - a.score);
    return actions.slice(0, 5);
  }

  getTypeEffectiveness(attackType, defenderTypes) {
    let multiplier = 1;
    const attackLower = attackType.toLowerCase();
    
    for (const defType of defenderTypes) {
      const defLower = defType.toLowerCase();
      const chart = TYPE_CHART[attackLower];
      if (chart && chart[defLower] !== undefined) {
        multiplier *= chart[defLower];
      }
    }
    
    return multiplier;
  }

  getPokemonStats(species) {
    // Better normalization to handle "-Paldea-Fire" etc.
    const normalized = species.replace(/[^a-zA-Z]/g, '');
    
    // Try exact match first
    if (POKEMON_STATS[species]) return POKEMON_STATS[species];
    
    // Try normalized match
    for (const key in POKEMON_STATS) {
      if (key.replace(/[^a-zA-Z]/g, '') === normalized) return POKEMON_STATS[key];
    }
    
    return null;
  }

  generateReasoning(moveData, effectiveness, statRatio, defender, potentialDamage, defenderStats) {
    // Reasoning is now generated by BattleCoach engine in action.rationale
    return "Use BattleCoach for reasoning";
  }

  getStatusLabel(status) {
    const labels = {
      brn: '🔥 BRN', par: '⚡ PAR', slp: '😴 SLP',
      frz: '❄️ FRZ', psn: '☠️ PSN', tox: '☠️ TOX'
    };
    return labels[status] || status.toUpperCase();
  }

  renderFieldConditions(state) {
    const field = state.field;
    if (!field) {
      // Handle missing field
      const statusContainer = document.getElementById('connectionStatus');
      if (statusContainer) {
          statusContainer.innerHTML = '⚔️ BATTLE ACTIVE';
          statusContainer.className = 'status-indicator battle';
      }
      return;
    }

    const fieldMetadata = state.fieldMetadata || {};
    
    // Render field conditions in the status bar instead of separate section
    const statusContainer = document.getElementById('connectionStatus');
    if (!statusContainer) return;
    
    const conditions = [];

    // Helper to get turn info
    const getDurationText = (key, maxTurns = 5) => {
        const meta = fieldMetadata[key];
        // Check if metadata matches the current active string
        if (meta && meta.startTurn && (meta.type === field[key] || field[key] === true)) {
             const turnsActive = (state.turn || 1) - meta.startTurn;
             const turnsLeft = Math.max(0, maxTurns - turnsActive);
             return ` (${turnsLeft} left)`;
        }
        return '';
    };

    if (field.weather) {
      const icons = { sun: '☀️', rain: '🌧️', sand: '🏜️', snow: '❄️' };
      const names = { sun: 'Sun', rain: 'Rain', sand: 'Sand', snow: 'Snow' };
      conditions.push(`<span class="field-badge">${icons[field.weather] || ''} ${names[field.weather] || field.weather}${getDurationText('weather')}</span>`);
    }
    
    if (field.terrain) {
      const icons = { electric: '⚡', grassy: '🌿', psychic: '🔮', misty: '🌫️' };
      const names = { electric: 'Electric', grassy: 'Grassy', psychic: 'Psychic', misty: 'Misty' };
      conditions.push(`<span class="field-badge">${icons[field.terrain] || ''} ${names[field.terrain] || field.terrain}${getDurationText('terrain')}</span>`);
    }
    
    if (field.trickRoom) {
         conditions.push(`<span class="field-badge">🔄 Trick Room${getDurationText('trickRoom')}</span>`);
    }

    if (field.tailwind) {
         conditions.push(`<span class="field-badge">💨 Tailwind${getDurationText('tailwind', 4)}</span>`);
    }
    
    // Update status bar with conditions if any, otherwise show battle active
    if (conditions.length > 0) {
      statusContainer.innerHTML = `<span class="battle-label">⚔️</span> ${conditions.join('')}`;
      statusContainer.className = 'status-indicator battle';
    } else {
      statusContainer.innerHTML = '⚔️ BATTLE ACTIVE';
      statusContainer.className = 'status-indicator battle';
    }
  }

  // Helper to format names: split camelCase/concatenated words and title case
  formatName(str) {
    if (!str) return '';
    // Insert space before capital letters
    let formatted = str.replace(/([a-z])([A-Z])/g, '$1 $2').replace(/([A-Z])([A-Z][a-z])/g, '$1 $2');
    
    // Split common concatenated terms (case-insensitive)
    const suffixes = ['vest', 'goggles', 'helmet', 'berry', 'herb', 'plate', 'band', 'specs', 'scarf', 'orb', 'sash', 'cloak', 'amulet', 'shield', 'dice', 'room', 'rock', 'stone', 'gem', 'lens', 'seed', 'tail', 'coat', 'fang', 'horn', 'claw', 'mask', 'punch', 'power', 'pulse', 'storm', 'beam', 'wind', 'wave', 'blitz', 'shot', 'turn', 'hand', 'combat', 'strikes', 'barrage', 'lance', 'drift', 'force', 'wind'];
    const prefixes = ['choice', 'safety', 'rocky', 'sitrus', 'covert', 'clear', 'ability', 'loaded', 'focus', 'life', 'expert', 'heavy', 'mental', 'wide', 'zoom', 'assault', 'super', 'hyper', 'mega', 'ultra', 'iron', 'armor', 'electric', 'grassy', 'misty', 'psychic', 'inner', 'multi', 'inner', 'swift', 'cloud', 'liquid', 'water', 'fake', 'flare', 'knock', 'parting', 'close', 'surging', 'astral', 'glacial', 'electro', 'expanding', 'helping'];
    
    let lowered = formatted.toLowerCase();
    
    // Corrections: Exact matches for common concatenated or weirdly spelled names
    const corrections = {
      'armortail': 'Armor Tail',
      'electricseed': 'Electric Seed',
      'grassyseed': 'Grassy Seed',
      'mistyseed': 'Misty Seed',
      'psychicseed': 'Psychic Seed',
      'assaultvest': 'Assault Vest',
      'choicescarf': 'Choice Scarf',
      'choicespecs': 'Choice Specs',
      'choiceband': 'Choice Band',
      'focussash': 'Focus Sash',
      'safetygoggles': 'Safety Goggles',
      'rockyhelmet': 'Rocky Helmet',
      'covertcloak': 'Covert Cloak',
      'clearamulet': 'Clear Amulet',
      'innerfocus': 'Inner Focus',
      'multiscale': 'Multi-Scale',
      'roughskin': 'Rough Skin',
      'fakeout': 'Fake Out',
      'uturn': 'U-turn',
      'vcreate': 'V-create',
      'knockoff': 'Knock Off',
      'flareblitz': 'Flare Blitz',
      'helpinghand': 'Helping Hand',
      'partingshot': 'Parting Shot'
    };
    
    if (corrections[lowered.replace(/\s+/g, '')]) return corrections[lowered.replace(/\s+/g, '')];

    // Check for common suffixes and inject space if missing
    for (const s of suffixes) {
      if (lowered.endsWith(s) && lowered.length > s.length && !lowered.includes(' ')) {
        formatted = lowered.slice(0, -s.length) + ' ' + s;
        break;
      }
    }
    
    // Check for common prefixes if still no space
    lowered = formatted.toLowerCase();
    if (!lowered.includes(' ')) {
      for (const p of prefixes) {
        if (lowered.startsWith(p) && lowered.length > p.length) {
          formatted = p + ' ' + lowered.slice(p.length);
          break;
        }
      }
    }
    
    return formatted
      .replace(/([a-z])([0-9])/gi, '$1 $2')
      .split(/[\s-]+/)
      .map(w => w.charAt(0).toUpperCase() + w.slice(1).toLowerCase())
      .join(' ');
  }
}

document.addEventListener('DOMContentLoaded', () => {
  const controller = new SidePanelController();
  controller.init();
});
