/**
 * Battle Coach - Type Effectiveness Module
 * Self-contained type chart utilities for Chrome extension
 */

const TYPES = [
  'normal', 'fire', 'water', 'electric', 'grass', 'ice',
  'fighting', 'poison', 'ground', 'flying', 'psychic', 'bug',
  'rock', 'ghost', 'dragon', 'dark', 'steel', 'fairy'
];

const TYPE_INDEX = {};
TYPES.forEach((t, i) => TYPE_INDEX[t] = i);

/**
 * Effectiveness chart: CHART[attacker][defender]
 * 0 = immune, 0.5 = resisted, 1 = neutral, 2 = super effective
 */
const CHART = [
  //NOR FIR WAT ELE GRA ICE FIG POI GRO FLY PSY BUG ROC GHO DRA DAR STE FAI
  [1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, .5,  0,  1,  1, .5,  1], // Normal
  [1, .5, .5,  1,  2,  2,  1,  1,  1,  1,  1,  2, .5,  1, .5,  1,  2,  1], // Fire
  [1,  2, .5,  1, .5,  1,  1,  1,  2,  1,  1,  1,  2,  1, .5,  1,  1,  1], // Water
  [1,  1,  2, .5, .5,  1,  1,  1,  0,  2,  1,  1,  1,  1, .5,  1,  1,  1], // Electric
  [1, .5,  2,  1, .5,  1,  1, .5,  2, .5,  1, .5,  2,  1, .5,  1, .5,  1], // Grass
  [1, .5, .5,  1,  2, .5,  1,  1,  2,  2,  1,  1,  1,  1,  2,  1, .5,  1], // Ice
  [2,  1,  1,  1,  1,  2,  1, .5,  1, .5, .5, .5,  2,  0,  1,  2,  2, .5], // Fighting
  [1,  1,  1,  1,  2,  1,  1, .5, .5,  1,  1,  1, .5, .5,  1,  1,  0,  2], // Poison
  [1,  2,  1,  2, .5,  1,  1,  2,  1,  0,  1, .5,  2,  1,  1,  1,  2,  1], // Ground
  [1,  1,  1, .5,  2,  1,  2,  1,  1,  1,  1,  2, .5,  1,  1,  1, .5,  1], // Flying
  [1,  1,  1,  1,  1,  1,  2,  2,  1,  1, .5,  1,  1,  1,  1,  0, .5,  1], // Psychic
  [1, .5,  1,  1,  2,  1, .5, .5,  1, .5,  2,  1,  1, .5,  1,  2, .5, .5], // Bug
  [1,  2,  1,  1,  1,  2, .5,  1, .5,  2,  1,  2,  1,  1,  1,  1, .5,  1], // Rock
  [0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  2,  1,  1,  2,  1, .5,  1,  1], // Ghost
  [1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  2,  1, .5,  0], // Dragon
  [1,  1,  1,  1,  1,  1, .5,  1,  1,  1,  2,  1,  1,  2,  1, .5,  1, .5], // Dark
  [1, .5, .5, .5,  1,  2,  1,  1,  1,  1,  1,  1,  2,  1,  1,  1, .5,  2], // Steel
  [1, .5,  1,  1,  1,  1,  2, .5,  1,  1,  1,  1,  1,  1,  2,  2, .5,  1], // Fairy
];

/**
 * Get type effectiveness multiplier
 */
function getTypeEffectiveness(attackType, defendType) {
  const atkIdx = TYPE_INDEX[attackType?.toLowerCase()];
  const defIdx = TYPE_INDEX[defendType?.toLowerCase()];
  
  if (atkIdx === undefined || defIdx === undefined) return 1;
  return CHART[atkIdx][defIdx];
}

/**
 * Get effectiveness against dual type
 */
function getEffectivenessVsDual(attackType, defendTypes) {
  if (!attackType || !defendTypes || defendTypes.length === 0) return 1;
  let mult = 1;
  for (const t of defendTypes) {
    mult *= getTypeEffectiveness(attackType, t);
  }
  return mult;
}

/**
 * Get all types super effective against a defender
 */
function getSuperEffectiveTypes(defendTypes) {
  if (!defendTypes || defendTypes.length === 0) return [];
  return TYPES.filter(t => getEffectivenessVsDual(t, defendTypes) >= 2);
}

/**
 * Get all types the defender resists
 */
function getResistTypes(defendTypes) {
  if (!defendTypes || defendTypes.length === 0) return [];
  return TYPES.filter(t => {
    const eff = getEffectivenessVsDual(t, defendTypes);
    return eff > 0 && eff < 1;
  });
}

/**
 * Get all types the defender is immune to
 */
function getImmuneTypes(defendTypes) {
  if (!defendTypes || defendTypes.length === 0) return [];
  return TYPES.filter(t => getEffectivenessVsDual(t, defendTypes) === 0);
}

/**
 * Check if a Pokemon has STAB on a move type
 */
function hasSTAB(moveType, pokemonTypes, teraType, teraActive) {
  const mt = moveType?.toLowerCase();
  if (teraActive && teraType) {
    return mt === teraType.toLowerCase() || pokemonTypes.some(t => t.toLowerCase() === mt);
  }
  return pokemonTypes.some(t => t.toLowerCase() === mt);
}

/**
 * Get STAB multiplier
 */
function getSTABMultiplier(moveType, pokemonTypes, teraType, teraActive) {
  const mt = moveType?.toLowerCase();
  const matchesOriginal = pokemonTypes.some(t => t.toLowerCase() === mt);
  
  if (!teraActive) {
    return matchesOriginal ? 1.5 : 1;
  }
  
  const matchesTera = teraType && mt === teraType.toLowerCase();
  
  if (matchesTera && matchesOriginal) return 2.0;  // Double STAB
  if (matchesTera || matchesOriginal) return 1.5;
  return 1;
}

// Expose to global scope for Chrome extension
window.TypeChart = {
  TYPES,
  getTypeEffectiveness,
  getEffectivenessVsDual,
  getSuperEffectiveTypes,
  getResistTypes,
  getImmuneTypes,
  hasSTAB,
  getSTABMultiplier
};
