/**
 * Speed Predictor - Speed Tier Classification for VGC Battles
 * Assigns speed tiers (S, A, B, C, D, F) based on base stats and modifiers
 */

// Speed tier thresholds - tighter bands at top for VGC meta
const SPEED_TIERS = [
  { tier: 'S', min: 130, label: 'Fastest' },      // Regieleki, Dragapult elite
  { tier: 'A', min: 110, label: 'Very Fast' },    // Flutter Mane, Tornadus
  { tier: 'B', min: 95, label: 'Fast' },          // Garchomp, Urshifu, Landorus-T
  { tier: 'C', min: 80, label: 'Average' },       // Rillaboom, Indeedee-F
  { tier: 'D', min: 50, label: 'Slow' },          // Incineroar, Iron Hands
  { tier: 'F', min: 0, label: 'Slowest' }         // Torkoal, Amoonguss, Dondozo
];

// Priority moves that let Pokemon move first regardless of speed
const HAS_PRIORITY_MOVES = new Set([
  'fakeout', 'quickguard', 'wideguard', 'extremespeed', 'feint', 
  'firstimpression', 'followme', 'ragepowder', 'aquajet', 'bulletpunch',
  'iceshard', 'jetpunch', 'machpunch', 'quickattack', 'shadowsneak',
  'suckerpunch', 'vacuumwave', 'watershuriken', 'accelerock', 'grassyglide'
]);

// Normalize name for lookup
function normalizeName(name) {
  return name.toLowerCase().replace(/[^a-z0-9]/g, '');
}

class SpeedPredictor {
  /**
   * Get base speed stat for a species
   */
  static getBaseSpeed(species) {
    // Use DataAdapter for base stats
    if (typeof DataAdapter !== 'undefined' && DataAdapter.isInitialized()) {
      const stats = DataAdapter.getBaseStats(species);
      if (stats?.spe) return stats.spe;
    }
    return 80; // Default fallback
  }

  /**
   * Check if species likely has a priority move in its common sets
   */
  static hasPriorityMove(species) {
    if (typeof SmogonDB === 'undefined' || !SmogonDB.isLoaded()) return false;
    
    const moves = SmogonDB.getAllMoves(species);
    if (!moves) return false;
    
    for (const [moveName, usage] of Object.entries(moves)) {
      const normalized = normalizeName(moveName);
      if (HAS_PRIORITY_MOVES.has(normalized) && usage > 10) {
        return true;
      }
    }
    return false;
  }

  /**
   * Check if species commonly uses Choice Scarf
   */
  static hasChoiceScarf(species) {
    if (typeof SmogonDB === 'undefined' || !SmogonDB.isLoaded()) return false;
    
    const data = SmogonDB.getPokemonData(species);
    if (!data?.items) return false;
    
    const scarfUsage = data.items['choicescarf'] || 0;
    return scarfUsage > 15; // Common if >15% usage
  }
  
  /**
   * Calculate effective speed for tier classification
   * @param {string} species - Pokemon species name
   * @param {boolean} hasTailwind - Whether Tailwind is active for this side
   * @returns {number} Effective base speed
   */
  static calculateEffectiveSpeed(species, hasTailwind = false) {
    let speed = this.getBaseSpeed(species);
    
    // Tailwind doubles speed
    if (hasTailwind) {
      speed *= 2;
    }
    
    // Choice Scarf adds 50% if commonly used
    if (this.hasChoiceScarf(species)) {
      speed *= 1.5;
    }
    
    return Math.round(speed);
  }
  
  /**
   * Get speed tier for a Pokemon
   * @param {string} species - Pokemon species
   * @param {boolean} hasTailwind - Whether side has Tailwind
   * @returns {{ tier: string, hasPriority: boolean, effectiveSpeed: number }}
   */
  static getSpeedTier(species, hasTailwind = false) {
    const effectiveSpeed = this.calculateEffectiveSpeed(species, hasTailwind);
    const hasPriority = this.hasPriorityMove(species);
    
    // Find matching tier
    let tier = 'F';
    for (const t of SPEED_TIERS) {
      if (effectiveSpeed >= t.min) {
        tier = t.tier;
        break;
      }
    }
    
    return { tier, hasPriority, effectiveSpeed };
  }
  
  /**
   * Get speed tiers for all active Pokemon
   * @param {Array} playerActive - Player's active Pokemon
   * @param {Array} opponentActive - Opponent's active Pokemon  
   * @param {Object} fieldState - Current field conditions
   * @returns {Object} Map of species -> tier info
   */
  static getAllSpeedTiers(playerActive, opponentActive, fieldState = {}) {
    const tiers = {};
    const playerHasTailwind = fieldState?.tailwind || false;
    const opponentHasTailwind = false; // Could track separately in future
    const trickRoom = fieldState?.trickRoom || false;
    
    // Process player Pokemon
    for (const p of playerActive) {
      if (!p?.species) continue;
      tiers[p.species] = this.getSpeedTier(p.species, playerHasTailwind);
      tiers[p.species].side = 'player';
      tiers[p.species].trickRoom = trickRoom;
    }
    
    // Process opponent Pokemon
    for (const p of opponentActive) {
      if (!p?.species) continue;
      tiers[p.species] = this.getSpeedTier(p.species, opponentHasTailwind);
      tiers[p.species].side = 'opponent';
      tiers[p.species].trickRoom = trickRoom;
    }
    
    return tiers;
  }
}

// Make available globally
if (typeof window !== 'undefined') {
  window.SpeedPredictor = SpeedPredictor;
}

if (typeof module !== 'undefined') {
  module.exports = SpeedPredictor;
}
