/**
 * SmogonDB - Query interface for VGC usage statistics
 * Provides move, item, ability, and tera usage data from Smogon stats
 */

// Inline the Smogon stats database (loaded at build time or bundled)
let SMOGON_DATA = null;

class SmogonDB {
  /**
   * Load the Smogon database from JSON
   * For Chrome extension, this should be called once at init
   */
  static async load() {
    if (SMOGON_DATA) return SMOGON_DATA;
    
    try {
      // In Chrome extension context, fetch from extension URL
      const url = chrome.runtime.getURL('data/smogon-stats.json');
      const response = await fetch(url);
      SMOGON_DATA = await response.json();
      console.log('🥋 SmogonDB loaded:', SMOGON_DATA.metadata?.pokemon_count, 'Pokémon');
      return SMOGON_DATA;
    } catch (e) {
      console.error('🥋 Failed to load SmogonDB:', e);
      return null;
    }
  }

  /**
   * Synchronous load for inline data (when bundled)
   */
  static loadSync(data) {
    SMOGON_DATA = data;
  }

  /**
   * Normalize species name for lookup (lowercase, hyphenated)
   */
  static normalizeSpecies(species) {
    if (!species) return '';
    return species
      .toLowerCase()
      .replace(/[^a-z0-9-]/g, '-')
      .replace(/-+/g, '-')
      .replace(/^-|-$/g, '');
  }

  /**
   * Format raw ID to readable name
   */
  static formatName(str) {
    if (!str) return '';
    const s = str.toLowerCase().replace(/[^a-z0-9]/g, '');
    
    const specials = {
      // Items
      'boosterenergy': 'Booster Energy',
      'lifeorb': 'Life Orb',
      'choiceband': 'Choice Band',
      'choicespecs': 'Choice Specs',
      'choicescarf': 'Choice Scarf',
      'assaultvest': 'Assault Vest',
      'focusash': 'Focus Sash',
      'sitrusberry': 'Sitrus Berry',
      'leftovers': 'Leftovers',
      'heavyboots': 'Heavy-Duty Boots',
      'clearamulet': 'Clear Amulet',
      'covertcloak': 'Covert Cloak',
      'loadeddice': 'Loaded Dice',
      'rockyhelmet': 'Rocky Helmet',
      'safetygoggles': 'Safety Goggles',
      'heatrock': 'Heat Rock',
      'damprock': 'Damp Rock',
      'icyrock': 'Icy Rock',
      'smoothrock': 'Smooth Rock',
      'whiteherb': 'White Herb',
      'mentalherb': 'Mental Herb',
      'powerherb': 'Power Herb',
      'mirrorherb': 'Mirror Herb',
      'weaknesspolicy': 'Weakness Policy',
      'ejectbutton': 'Eject Button',
      'ejectpack': 'Eject Pack',
      'redcard': 'Red Card',
      'roomservice': 'Room Service',

      // Abilities
      'thermalexchange': 'Thermal Exchange',
      'protosynthesis': 'Protosynthesis',
      'quarkdrive': 'Quark Drive',
      'swordofruin': 'Sword of Ruin',
      'beadsofruin': 'Beads of Ruin',
      'tabletsofruin': 'Tablets of Ruin',
      'vesselofruin': 'Vessel of Ruin',
      'innerfocus': 'Inner Focus',
      'intimidate': 'Intimidate',
      'regenerator': 'Regenerator',
      'unseenfist': 'Unseen Fist',
      'libero': 'Libero',
      'protean': 'Protean',
      'defeatist': 'Defeatist',
      'clearbody': 'Clear Body',
      'roughskin': 'Rough Skin',
      'ironbarbs': 'Iron Barbs',
      'hugepower': 'Huge Power',
      'purepower': 'Pure Power',
      'moldbreaker': 'Mold Breaker',
      'levitate': 'Levitate',
      'multiscale': 'Multiscale',
      'shadowshield': 'Shadow Shield',
      'fullmetalbody': 'Full Metal Body',
      'prismarmor': 'Prism Armor',
      'neuralsforce': 'Neuroforce',
      'supremeoverlord': 'Supreme Overlord',
      'sharpness': 'Sharpness',
      'hadronengine': 'Hadron Engine',
      'orichalcumpulse': 'Orichalcum Pulse',
      'wellbakedbody': 'Well-Baked Body',
      'goodasgold': 'Good as Gold',
      'purifyingsalt': 'Purifying Salt',
      'toxicdebris': 'Toxic Debris',
      'earthaneater': 'Earth Eater',
      'commander': 'Commander',
      'embodyaspect': 'Embody Aspect',
      'hospitality': 'Hospitality',
      'mindseye': 'Minds Eye',
      'toxicchain': 'Toxic Chain',
      'supersweetaroma': 'Super Sweet Aroma',
      'poisonpuppeteer': 'Poison Puppeteer',
      'storageup': 'Storage Up',
      'zerotohero': 'Zero to Hero',
      'paldeanform': 'Paldean Form',
      'asone': 'As One',
      'asonespectrier': 'As One (Spectrier)',
      'asoneglastrier': 'As One (Glastrier)',
      'chillingneigh': 'Chilling Neigh',
      'grimneigh': 'Grim Neigh',
      'friendguard': 'Friend Guard',
      'widelens': 'Wide Lens',
      'technician': 'Technician',
      'magicaltorque': 'Magical Torque',
      'combattorque': 'Combat Torque',
      'blazingtorque': 'Blazing Torque',
      'noxioustorque': 'Noxious Torque',
      'wickedtorque': 'Wicked Torque',
      'populationbomb': 'Population Bomb'
    };
    
    if (specials[s]) return specials[s];

    // Generic formatting: Title Case words
    // If it's a single word slug (no separators), just capitalize first letter
    if (!str.includes(' ') && !str.includes('-') && !str.includes('_')) {
      return str.charAt(0).toUpperCase() + str.slice(1).toLowerCase();
    }

    return str
      .replace(/([a-z])([A-Z])/g, '$1 $2')
      .replace(/[^a-zA-Z0-9 ]+/g, ' ')
      .trim()
      .split(/\s+/)
      .map(word => word.charAt(0).toUpperCase() + word.slice(1).toLowerCase())
      .join(' ');
  }

  /**
   * Mapping for species that should default to a specific form in VGC 
   * when the base name is detected (e.g. "Ninetales" -> "Ninetales-Alola")
   */
  static COMPETITIVE_FORM_MAP = {
    'ninetales': 'ninetales-alola',
    'arcanine': 'arcanine-hisui',
    'raichu': 'raichu-alola',
    'persian': 'persian-alola',
    'muk': 'muk-alola',
    'marowak': 'marowak-alola'
  };

  /**
   * Get Pokemon data from database
   */
  static getPokemonData(species) {
    if (!SMOGON_DATA?.pokemon) return null;
    
    let normalized = this.normalizeSpecies(species);
    
    // VGC Redirection: If we see a base name that is almost always a regional form in VGC
    if (this.COMPETITIVE_FORM_MAP[normalized]) {
      normalized = this.COMPETITIVE_FORM_MAP[normalized];
    }

    // Try exact match first
    if (SMOGON_DATA.pokemon[normalized]) {
      return SMOGON_DATA.pokemon[normalized];
    }
    
    // Try without form suffix
    const baseName = normalized.split('-')[0];
    if (SMOGON_DATA.pokemon[baseName]) {
      return SMOGON_DATA.pokemon[baseName];
    }
    
    // Try fuzzy match
    for (const key of Object.keys(SMOGON_DATA.pokemon)) {
      if (key.startsWith(baseName)) {
        return SMOGON_DATA.pokemon[key];
      }
    }
    
    return null;
  }

  /**
   * Get move usage percentage for a Pokemon (0-100)
   * Returns 0 if move not in database
   */
  static getMoveUsage(species, moveName) {
    const data = this.getPokemonData(species);
    if (!data?.moves) return 0;
    
    // Normalize move name: lowercase, remove spaces and special chars
    const normalizedMove = moveName.toLowerCase().replace(/[^a-z0-9]/g, '');
    
    // Try exact normalized match
    if (data.moves[normalizedMove] !== undefined) {
      return data.moves[normalizedMove];
    }
    
    // Try matching against normalized versions of database keys
    for (const [move, usage] of Object.entries(data.moves)) {
      const dbMoveNormalized = move.toLowerCase().replace(/[^a-z0-9]/g, '');
      if (dbMoveNormalized === normalizedMove) {
        return usage;
      }
    }
    
    return 0;
  }

  /**
   * Get all moves with usage data for a Pokemon
   */
  static getAllMoves(species) {
    const data = this.getPokemonData(species);
    return data?.moves || {};
  }

  /**
   * Get top item for a Pokemon
   * @returns {{ name: string, usage: number } | null}
   */
  static getTopItem(species) {
    const data = this.getPokemonData(species);
    if (!data?.items) return null;
    
    let topItem = null;
    let topUsage = 0;
    
    for (const [item, usage] of Object.entries(data.items)) {
      if (usage > topUsage) {
        topItem = item;
        topUsage = usage;
      }
    }
    
    return topItem ? { name: this.formatName(topItem), usage: topUsage } : null;
  }

  /**
   * Get top ability for a Pokemon
   * @returns {{ name: string, usage: number } | null}
   */
  static getTopAbility(species) {
    const data = this.getPokemonData(species);
    if (!data?.abilities) return null;
    
    let topAbility = null;
    let topUsage = 0;
    
    for (const [ability, usage] of Object.entries(data.abilities)) {
      if (usage > topUsage) {
        topAbility = ability;
        topUsage = usage;
      }
    }
    
    return topAbility ? { name: this.formatName(topAbility), usage: topUsage } : null;
  }

  /**
   * Get tera type usage percentage (0-100)
   */
  static getTeraUsage(species, teraType) {
    const data = this.getPokemonData(species);
    if (!data?.tera_types) return 0;
    
    // Normalize type name
    const normalizedType = teraType.charAt(0).toUpperCase() + teraType.slice(1).toLowerCase();
    
    return data.tera_types[normalizedType] || 0;
  }

  /**
   * Get all tera types with usage for a Pokemon
   */
  static getAllTeraTypes(species) {
    const data = this.getPokemonData(species);
    return data?.tera_types || {};
  }

  /**
   * Get Pokemon usage rate (how common in the meta)
   */
  static getUsageRate(species) {
    const data = this.getPokemonData(species);
    return data?.usage_rate || 0;
  }

  /**
   * Check if database is loaded
   */
  static isLoaded() {
    return SMOGON_DATA !== null;
  }

  /**
   * Get metadata about the database
   */
  static getMetadata() {
    return SMOGON_DATA?.metadata || null;
  }
}

// Make available globally
if (typeof window !== 'undefined') {
  window.SmogonDB = SmogonDB;
}
