/**
 * Battle Coach - Role System
 * Dynamic role detection based on revealed info and game state
 */

const ROLES = {
  PHYSICAL_ATTACKER: {
    name: 'Physical Attacker',
    fears: ['burn', 'intimidate'],
    immunities: {
      burn: ['guts', 'water veil', 'thermal exchange'],
      intimidate: ['defiant', 'competitive', 'clear body', 'white smoke', 
                   'full metal body', 'inner focus', 'oblivious', 'own tempo']
    }
  },
  SPECIAL_ATTACKER: {
    name: 'Special Attacker',
    fears: ['special_wall'],
    immunities: {}
  },
  TANK: {
    name: 'Tank',
    fears: ['toxic', 'passive_damage'],
    immunities: { toxic: ['immunity', 'poison heal', 'magic guard'] }
  },
  WALL: {
    name: 'Wall',
    fears: ['taunt', 'toxic', 'setup_sweeper'],
    immunities: { taunt: ['oblivious', 'aroma veil'] }
  },
  SPEED_CONTROL: {
    name: 'Speed Control',
    fears: ['taunt', 'fake_out'],
    immunities: {
      taunt: ['oblivious', 'own tempo'],
      fake_out: ['inner focus', 'covert cloak']
    }
  },
  SUPPORT: {
    name: 'Support',
    fears: ['taunt', 'encore'],
    immunities: { taunt: ['oblivious', 'own tempo'] }
  },
  INTIMIDATE_PIVOT: {
    name: 'Intimidate Pivot',
    fears: ['defiant_trigger', 'competitive_trigger'],
    immunities: {}
  },
  PIVOT: {
    name: 'Pivot',
    fears: ['entry_hazards', 'trapping'],
    immunities: {}
  },
  TRICK_ROOM_SETTER: {
    name: 'Trick Room Setter',
    fears: ['taunt', 'fake_out', 'ko_before_move'],
    immunities: {
      taunt: ['oblivious'],
      fake_out: ['inner focus', 'covert cloak']
    }
  },
  TRICK_ROOM_SWEEPER: {
    name: 'Trick Room Sweeper',
    fears: ['trick_room_ends'],
    immunities: {}
  },
  WEATHER_SETTER: {
    name: 'Weather Setter',
    fears: ['weather_override'],
    immunities: {}
  },
  WEATHER_ABUSER: {
    name: 'Weather Abuser',
    fears: ['weather_change'],
    immunities: {}
  },
  TERRAIN_SETTER: {
    name: 'Terrain Setter',
    fears: ['terrain_override'],
    immunities: {}
  },
  REDIRECTION: {
    name: 'Redirection',
    fears: ['fake_out', 'ko'],
    immunities: { fake_out: ['inner focus'] }
  },
  SETUP_SWEEPER: {
    name: 'Setup Sweeper',
    fears: ['taunt', 'encore', 'priority', 'ko_before_setup'],
    immunities: { taunt: ['oblivious'] }
  }
};

// Move categorization
const PHYSICAL_MOVES = [
  'earthquake', 'rock slide', 'close combat', 'flare blitz', 'knock off',
  'u-turn', 'wood hammer', 'drain punch', 'wild charge', 'heavy slam',
  'wicked blow', 'surging strikes', 'extreme speed', 'aqua jet', 'bullet punch',
  'ice shard', 'mach punch', 'shadow sneak', 'sucker punch', 'grassy glide',
  'brave bird', 'iron head', 'crunch', 'play rough', 'sacred sword'
];

const SPECIAL_MOVES = [
  'shadow ball', 'moonblast', 'dazzling gleam', 'heat wave', 'draco meteor',
  'thunderbolt', 'ice beam', 'energy ball', 'psychic', 'dark pulse',
  'flamethrower', 'hydro pump', 'make it rain', 'bleakwind storm', 'muddy water',
  'sludge bomb', 'flash cannon', 'aura sphere', 'focus blast'
];

const SPEED_CONTROL_MOVES = [
  'tailwind', 'trick room', 'thunder wave', 'icy wind', 'electroweb',
  'scary face', 'string shot', 'cotton spore', 'glaciate'
];

const SUPPORT_MOVES = [
  'follow me', 'rage powder', 'helping hand', 'fake out', 'taunt', 'encore',
  'reflect', 'light screen', 'safeguard', 'wide guard', 'quick guard',
  'ally switch', 'heal pulse', 'life dew', 'pollen puff', 'will-o-wisp',
  'spore', 'sleep powder', 'parting shot'
];

const SETUP_MOVES = [
  'swords dance', 'dragon dance', 'nasty plot', 'calm mind', 'quiver dance',
  'shell smash', 'belly drum', 'bulk up', 'coil', 'iron defense', 'amnesia',
  'agility', 'rock polish', 'autotomize', 'shift gear', 'geomancy'
];

const WIN_CONDITION_POKEMON = [
  'flutter mane', 'chi-yu', 'urshifu', 'urshifu-rapid-strike',
  'iron hands', 'dragonite', 'kingambit', 'iron bundle',
  'garchomp', 'gholdengo', 'koraidon', 'miraidon', 'calyrex-shadow',
  'calyrex-ice', 'zacian', 'kyogre'
];

function hasPhysicalMoves(moves) {
  return moves.some(m => PHYSICAL_MOVES.includes(m.toLowerCase()));
}

function hasSpecialMoves(moves) {
  return moves.some(m => SPECIAL_MOVES.includes(m.toLowerCase()));
}

function hasSpeedControl(moves) {
  return moves.some(m => SPEED_CONTROL_MOVES.includes(m.toLowerCase()));
}

function hasSupportMoves(moves) {
  return moves.some(m => SUPPORT_MOVES.includes(m.toLowerCase()));
}

function hasSetupMoves(moves) {
  return moves.some(m => SETUP_MOVES.includes(m.toLowerCase()));
}

/**
 * Detect active roles for a Pokemon
 */
function detectRoles(pokemon, gameState) {
  const roles = new Set();
  const moves = (pokemon.moves || []).map(m => m.toLowerCase());
  const ability = (pokemon.ability || '').toLowerCase();
  const likelyMoves = (pokemon.likelyMoves || []).map(m => m.toLowerCase());
  const allMoves = [...new Set([...moves, ...likelyMoves])];
  
  if (hasPhysicalMoves(allMoves)) roles.add('PHYSICAL_ATTACKER');
  if (hasSpecialMoves(allMoves)) roles.add('SPECIAL_ATTACKER');
  if (hasSpeedControl(allMoves)) roles.add('SPEED_CONTROL');
  
  if (allMoves.includes('trick room')) {
    roles.add('TRICK_ROOM_SETTER');
    if (gameState?.field?.trickRoom) roles.add('TRICK_ROOM_SWEEPER');
  }
  
  if (hasSupportMoves(allMoves)) roles.add('SUPPORT');
  if (ability === 'intimidate') roles.add('INTIMIDATE_PIVOT');
  
  // Pivot detection (U-turn/Volt Switch/Parting Shot/Flip Turn)
  if (allMoves.some(m => ['u-turn', 'volt switch', 'flip turn', 'parting shot'].includes(m))) {
    if (ability !== 'intimidate') roles.add('PIVOT');
  }

  // TANK/WALL detection based on stats and moves
  // Since we don't have stats here easily without async, we rely on move heuristics
  const recoveryMoves = ['recover', 'roost', 'slack off', 'soft-boiled', 'milk drink', 'moonlight', 'morning sun', 'synthesis', 'wish'];
  const hasRecovery = allMoves.some(m => recoveryMoves.includes(m));
  const hasDefense = allMoves.some(m => ['iron defense', 'amnesia', 'cotton guard', 'acid armor'].includes(m));
  
  if (hasRecovery && (hasDefense || hasSupportMoves(allMoves))) {
     roles.add('WALL');
  } else if (hasRecovery && (hasPhysicalMoves(allMoves) || hasSpecialMoves(allMoves))) {
     roles.add('TANK'); // Attack + Recovery
  }
  
  if (['drought', 'drizzle', 'sand stream', 'snow warning', 'orichalcum pulse', 
       'desolate land', 'primordial sea'].includes(ability)) {
    roles.add('WEATHER_SETTER');
  }
  
  if (['chlorophyll', 'swift swim', 'sand rush', 'slush rush', 'solar power',
       'protosynthesis'].includes(ability)) {
    roles.add('WEATHER_ABUSER');
  }
  
  if (['grassy surge', 'electric surge', 'psychic surge', 'misty surge'].includes(ability)) {
    roles.add('TERRAIN_SETTER');
  }
  
  if (allMoves.includes('follow me') || allMoves.includes('rage powder') ||
      ['storm drain', 'lightning rod'].includes(ability)) {
    roles.add('REDIRECTION');
  }
  
  if (hasSetupMoves(allMoves)) roles.add('SETUP_SWEEPER');
  
  if (roles.size === 0) roles.add('PHYSICAL_ATTACKER');
  
  return Array.from(roles);
}

/**
 * Check if a Pokemon's role fears a specific threat
 */
function roleFearsThreat(role, threat, ability, item) {
  const roleDef = ROLES[role];
  if (!roleDef || !roleDef.fears.includes(threat)) return false;
  
  const immunities = roleDef.immunities[threat] || [];
  const abilityLower = (ability || '').toLowerCase();
  const itemLower = (item || '').toLowerCase();
  
  for (const immunity of immunities) {
    if (abilityLower.includes(immunity.toLowerCase())) return false;
    if (immunity === 'covert cloak' && itemLower === 'covert cloak') return false;
  }
  
  return true;
}

/**
 * Check if a Pokemon is likely a win condition
 */
function isWinCondition(name) {
  return WIN_CONDITION_POKEMON.includes(name.toLowerCase());
}

// Expose to global scope
window.RoleSystem = {
  ROLES,
  detectRoles,
  roleFearsThreat,
  isWinCondition
};
