/**
 * Battle Coach - Move Database
 * Essential move data for damage calculations and strategic evaluation
 */

const MOVES = {
  // PRIORITY MOVES
  'fake out': {
    name: 'Fake Out', type: 'normal', category: 'physical', basePower: 40,
    accuracy: 100, priority: 3, target: 'single', flags: ['contact'],
    effect: { flinch: 100 }
  },
  'extreme speed': {
    name: 'Extreme Speed', type: 'normal', category: 'physical', basePower: 80,
    accuracy: 100, priority: 2, target: 'single', flags: ['contact']
  },
  'aqua jet': {
    name: 'Aqua Jet', type: 'water', category: 'physical', basePower: 40,
    accuracy: 100, priority: 1, target: 'single', flags: ['contact']
  },
  'bullet punch': {
    name: 'Bullet Punch', type: 'steel', category: 'physical', basePower: 40,
    accuracy: 100, priority: 1, target: 'single', flags: ['contact', 'punch']
  },
  'ice shard': {
    name: 'Ice Shard', type: 'ice', category: 'physical', basePower: 40,
    accuracy: 100, priority: 1, target: 'single', flags: ['contact']
  },
  'mach punch': {
    name: 'Mach Punch', type: 'fighting', category: 'physical', basePower: 40,
    accuracy: 100, priority: 1, target: 'single', flags: ['contact', 'punch']
  },
  'quick attack': {
    name: 'Quick Attack', type: 'normal', category: 'physical', basePower: 40,
    accuracy: 100, priority: 1, target: 'single', flags: ['contact']
  },
  'shadow sneak': {
    name: 'Shadow Sneak', type: 'ghost', category: 'physical', basePower: 40,
    accuracy: 100, priority: 1, target: 'single', flags: ['contact']
  },
  'sucker punch': {
    name: 'Sucker Punch', type: 'dark', category: 'physical', basePower: 70,
    accuracy: 100, priority: 1, target: 'single', flags: ['contact']
  },
  'grassy glide': {
    name: 'Grassy Glide', type: 'grass', category: 'physical', basePower: 55,
    accuracy: 100, priority: 0, target: 'single', flags: ['contact']
  },
  
  // PROTECT MOVES
  'protect': {
    name: 'Protect', type: 'normal', category: 'status', basePower: 0,
    accuracy: 101, priority: 4, target: 'self', flags: []
  },
  'detect': {
    name: 'Detect', type: 'fighting', category: 'status', basePower: 0,
    accuracy: 101, priority: 4, target: 'self', flags: []
  },
  'wide guard': {
    name: 'Wide Guard', type: 'rock', category: 'status', basePower: 0,
    accuracy: 101, priority: 3, target: 'field', flags: []
  },
  'quick guard': {
    name: 'Quick Guard', type: 'fighting', category: 'status', basePower: 0,
    accuracy: 101, priority: 3, target: 'field', flags: []
  },
  
  // SPREAD MOVES
  'earthquake': {
    name: 'Earthquake', type: 'ground', category: 'physical', basePower: 100,
    accuracy: 100, priority: 0, target: 'allAdjacent', flags: []
  },
  'rock slide': {
    name: 'Rock Slide', type: 'rock', category: 'physical', basePower: 75,
    accuracy: 90, priority: 0, target: 'allAdjacentFoes', flags: [],
    effect: { flinch: 30 }
  },
  'dazzling gleam': {
    name: 'Dazzling Gleam', type: 'fairy', category: 'special', basePower: 80,
    accuracy: 100, priority: 0, target: 'allAdjacentFoes', flags: []
  },
  'heat wave': {
    name: 'Heat Wave', type: 'fire', category: 'special', basePower: 95,
    accuracy: 90, priority: 0, target: 'allAdjacentFoes', flags: [],
    effect: { burn: 10 }
  },
  'icy wind': {
    name: 'Icy Wind', type: 'ice', category: 'special', basePower: 55,
    accuracy: 95, priority: 0, target: 'allAdjacentFoes', flags: ['wind'],
    effect: { speedDrop: 100 }
  },
  'electroweb': {
    name: 'Electroweb', type: 'electric', category: 'special', basePower: 55,
    accuracy: 95, priority: 0, target: 'allAdjacentFoes', flags: [],
    effect: { speedDrop: 100 }
  },
  'snarl': {
    name: 'Snarl', type: 'dark', category: 'special', basePower: 55,
    accuracy: 95, priority: 0, target: 'allAdjacentFoes', flags: ['sound'],
    effect: { spatkDrop: 100 }
  },
  'muddy water': {
    name: 'Muddy Water', type: 'water', category: 'special', basePower: 90,
    accuracy: 85, priority: 0, target: 'allAdjacentFoes', flags: []
  },
  'bleakwind storm': {
    name: 'Bleakwind Storm', type: 'flying', category: 'special', basePower: 100,
    accuracy: 80, priority: 0, target: 'allAdjacentFoes', flags: ['wind']
  },
  'make it rain': {
    name: 'Make It Rain', type: 'steel', category: 'special', basePower: 120,
    accuracy: 100, priority: 0, target: 'allAdjacentFoes', flags: []
  },
  'astral barrage': {
    name: 'Astral Barrage', type: 'ghost', category: 'special', basePower: 120,
    accuracy: 100, priority: 0, target: 'allAdjacentFoes', flags: []
  },
  'glacial lance': {
    name: 'Glacial Lance', type: 'ice', category: 'physical', basePower: 120,
    accuracy: 100, priority: 0, target: 'allAdjacentFoes', flags: []
  },
  
  // SINGLE-TARGET ATTACKS
  'close combat': {
    name: 'Close Combat', type: 'fighting', category: 'physical', basePower: 120,
    accuracy: 100, priority: 0, target: 'single', flags: ['contact']
  },
  'flare blitz': {
    name: 'Flare Blitz', type: 'fire', category: 'physical', basePower: 120,
    accuracy: 100, priority: 0, target: 'single', flags: ['contact']
  },
  'knock off': {
    name: 'Knock Off', type: 'dark', category: 'physical', basePower: 65,
    accuracy: 100, priority: 0, target: 'single', flags: ['contact']
  },
  'u-turn': {
    name: 'U-turn', type: 'bug', category: 'physical', basePower: 70,
    accuracy: 100, priority: 0, target: 'single', flags: ['contact'],
    effect: { pivot: true }
  },
  'volt switch': {
    name: 'Volt Switch', type: 'electric', category: 'special', basePower: 70,
    accuracy: 100, priority: 0, target: 'single', flags: [],
    effect: { pivot: true }
  },
  'parting shot': {
    name: 'Parting Shot', type: 'dark', category: 'status', basePower: 0,
    accuracy: 100, priority: 0, target: 'single', flags: ['sound'],
    effect: { pivot: true }
  },
  'shadow ball': {
    name: 'Shadow Ball', type: 'ghost', category: 'special', basePower: 80,
    accuracy: 100, priority: 0, target: 'single', flags: ['bullet']
  },
  'moonblast': {
    name: 'Moonblast', type: 'fairy', category: 'special', basePower: 95,
    accuracy: 100, priority: 0, target: 'single', flags: []
  },
  'draco meteor': {
    name: 'Draco Meteor', type: 'dragon', category: 'special', basePower: 130,
    accuracy: 90, priority: 0, target: 'single', flags: []
  },
  'wicked blow': {
    name: 'Wicked Blow', type: 'dark', category: 'physical', basePower: 75,
    accuracy: 100, priority: 0, target: 'single', flags: ['contact', 'punch']
  },
  'surging strikes': {
    name: 'Surging Strikes', type: 'water', category: 'physical', basePower: 25,
    accuracy: 100, priority: 0, target: 'single', flags: ['contact', 'punch']
  },
  'wood hammer': {
    name: 'Wood Hammer', type: 'grass', category: 'physical', basePower: 120,
    accuracy: 100, priority: 0, target: 'single', flags: ['contact']
  },
  'drain punch': {
    name: 'Drain Punch', type: 'fighting', category: 'physical', basePower: 75,
    accuracy: 100, priority: 0, target: 'single', flags: ['contact', 'punch']
  },
  'high horsepower': {
    name: 'High Horsepower', type: 'ground', category: 'physical', basePower: 95,
    accuracy: 95, priority: 0, target: 'single', flags: ['contact']
  },
  'electro drift': {
    name: 'Electro Drift', type: 'electric', category: 'special', basePower: 100,
    accuracy: 100, priority: 0, target: 'single', flags: ['contact']
  },
  
  // SPEED CONTROL
  'tailwind': {
    name: 'Tailwind', type: 'flying', category: 'status', basePower: 0,
    accuracy: 101, priority: 0, target: 'field', flags: ['wind'],
    effect: { tailwind: 4 }
  },
  'trick room': {
    name: 'Trick Room', type: 'psychic', category: 'status', basePower: 0,
    accuracy: 101, priority: -7, target: 'field', flags: [],
    effect: { trickRoom: 5 }
  },
  'thunder wave': {
    name: 'Thunder Wave', type: 'electric', category: 'status', basePower: 0,
    accuracy: 90, priority: 0, target: 'single', flags: [],
    effect: { paralysis: 100 }
  },
  
  // SUPPORT MOVES
  'follow me': {
    name: 'Follow Me', type: 'normal', category: 'status', basePower: 0,
    accuracy: 101, priority: 2, target: 'self', flags: [],
    effect: { redirect: true }
  },
  'rage powder': {
    name: 'Rage Powder', type: 'bug', category: 'status', basePower: 0,
    accuracy: 101, priority: 2, target: 'self', flags: ['powder'],
    effect: { redirect: true }
  },
  'helping hand': {
    name: 'Helping Hand', type: 'normal', category: 'status', basePower: 0,
    accuracy: 101, priority: 5, target: 'ally', flags: [],
    effect: { helpingHand: true }
  },
  'taunt': {
    name: 'Taunt', type: 'dark', category: 'status', basePower: 0,
    accuracy: 100, priority: 0, target: 'single', flags: []
  },
  'encore': {
    name: 'Encore', type: 'normal', category: 'status', basePower: 0,
    accuracy: 100, priority: 0, target: 'single', flags: []
  },
  'will-o-wisp': {
    name: 'Will-O-Wisp', type: 'fire', category: 'status', basePower: 0,
    accuracy: 85, priority: 0, target: 'single', flags: []
  },
  'spore': {
    name: 'Spore', type: 'grass', category: 'status', basePower: 0,
    accuracy: 100, priority: 0, target: 'single', flags: ['powder']
  },
  'reflect': {
    name: 'Reflect', type: 'psychic', category: 'status', basePower: 0,
    accuracy: 101, priority: 0, target: 'field', flags: []
  },
  'light screen': {
    name: 'Light Screen', type: 'psychic', category: 'status', basePower: 0,
    accuracy: 101, priority: 0, target: 'field', flags: []
  },
  
  // SETUP MOVES
  'swords dance': {
    name: 'Swords Dance', type: 'normal', category: 'status', basePower: 0,
    accuracy: 101, priority: 0, target: 'self', flags: [],
    effect: { selfAtkBoost: 2 }
  },
  'dragon dance': {
    name: 'Dragon Dance', type: 'dragon', category: 'status', basePower: 0,
    accuracy: 101, priority: 0, target: 'self', flags: [],
    effect: { selfAtkBoost: 1, selfSpeBoost: 1 }
  },
  'nasty plot': {
    name: 'Nasty Plot', type: 'dark', category: 'status', basePower: 0,
    accuracy: 101, priority: 0, target: 'self', flags: [],
    effect: { selfSpAtkBoost: 2 }
  },
  'calm mind': {
    name: 'Calm Mind', type: 'psychic', category: 'status', basePower: 0,
    accuracy: 101, priority: 0, target: 'self', flags: [],
    effect: { selfSpAtkBoost: 1, selfSpDefBoost: 1 }
  },
  'belly drum': {
    name: 'Belly Drum', type: 'normal', category: 'status', basePower: 0,
    accuracy: 101, priority: 0, target: 'self', flags: [],
    effect: { selfAtkBoost: 6, selfDamage: 50 }
  }
};

function getMove(moveName) {
  if (!moveName) return null;
  return MOVES[moveName.toLowerCase()] || null;
}

function isSpreadMove(moveName) {
  const move = getMove(moveName);
  if (!move) return false;
  return ['allAdjacent', 'allAdjacentFoes'].includes(move.target);
}

function isPriorityMove(moveName) {
  const move = getMove(moveName);
  return move ? move.priority > 0 : false;
}

function isProtectMove(moveName) {
  const name = moveName?.toLowerCase();
  return ['protect', 'detect', 'spiky shield', 'baneful bunker', "king's shield", 
          'obstruct', 'silk trap', 'burning bulwark'].includes(name);
}

function isSpeedControl(moveName) {
  const move = getMove(moveName);
  if (!move) return false;
  const name = moveName.toLowerCase();
  if (['tailwind', 'trick room', 'thunder wave', 'icy wind', 'electroweb', 
       'scary face', 'cotton spore', 'string shot'].includes(name)) {
    return true;
  }
  return move.effect?.speedDrop || move.effect?.paralysis;
}

function isSetupMove(moveName) {
  const move = getMove(moveName);
  if (!move) return false;
  const effect = move.effect || {};
  return effect.selfAtkBoost || effect.selfSpAtkBoost || effect.selfSpeBoost ||
         effect.selfDefBoost || effect.selfSpDefBoost;
}

function isPivotMove(moveName) {
  const move = getMove(moveName);
  return move?.effect?.pivot === true;
}

function isRedirectMove(moveName) {
  const move = getMove(moveName);
  return move?.effect?.redirect === true;
}

// Expose to global scope
window.MoveData = {
  MOVES,
  getMove,
  isSpreadMove,
  isPriorityMove,
  isProtectMove,
  isSpeedControl,
  isSetupMove,
  isPivotMove,
  isRedirectMove
};
