/**
 * Battle Coach - Scoring Engine
 * Core algorithm: SCORE = Tactical × Role × Blindspot × Meta
 */

// Dependencies from other modules (loaded as window globals)
const getTypeChart = () => window.TypeChart;
const getMoveData = () => window.MoveData;
const getRoleSystem = () => window.RoleSystem;
const getBlindspotWeights = () => window.BlindspotWeights;
const getDataAdapter = () => window.DataAdapter;

let initialized = false;

// =============================================================================
// INITIALIZATION
// =============================================================================

/**
 * Initialize the Battle Coach engine
 */
function initialize(databases) {
  const DataAdapter = getDataAdapter();
  if (DataAdapter) {
    DataAdapter.initializeData(databases);
    initialized = true;
    console.log('[BattleCoach] Engine initialized');
  } else {
    console.error('[BattleCoach] DataAdapter not loaded');
  }
}

/**
 * Check if engine is ready
 */
function isReady() {
  return initialized && getDataAdapter()?.isInitialized();
}

// =============================================================================
// MAIN ANALYSIS
// =============================================================================

/**
 * Analyze the current game state and return recommendations
 */
function analyze(gameState) {
  if (!isReady()) {
    console.warn('[BattleCoach] Not initialized');
    return { error: 'Not initialized', recommendations: [] };
  }
  
  const recommendations = getAllRecommendations(gameState);
  const speedOrder = calculateSpeedOrder(gameState);
  
  return {
    turn: gameState.turn,
    recommendations,
    speedOrder,
    timestamp: Date.now()
  };
}

/**
 * Get recommendations for all active player slots
 */
function getAllRecommendations(gameState) {
  const results = [];
  const playerActive = gameState.playerActive || [];
  
  for (let i = 0; i < playerActive.length; i++) {
    const pokemon = playerActive[i];
    if (!pokemon || pokemon.hp <= 0) continue;
    
    const slotActions = analyzeSlot(pokemon, i, gameState);
    results.push({
      slot: i,
      pokemon: pokemon.name,
      actions: slotActions
    });
  }
  
  return results;
}

// =============================================================================
// SLOT ANALYSIS
// =============================================================================

/**
 * Analyze a single Pokemon slot and score all possible actions
 */
function analyzeSlot(pokemon, slotIndex, gameState) {
  const DataAdapter = getDataAdapter();
  const enrichedPokemon = DataAdapter.inferUnknowns(pokemon);
  const enrichedState = enrichState(gameState);
  
  // Enumerate all possible actions
  const actions = enumerateActions(enrichedPokemon, slotIndex, enrichedState);
  
  // Score each action
  const scoredActions = actions.map(action => {
    return scoreAction(action, enrichedPokemon, enrichedState);
  });
  
  // Sort by score descending
  scoredActions.sort((a, b) => b.score - a.score);
  
  // Return top 5
  return scoredActions.slice(0, 5);
}

/**
 * Enrich game state with inferred data
 */
function enrichState(gameState) {
  const DataAdapter = getDataAdapter();
  const enriched = { ...gameState };
  
  // Enrich opponent active Pokemon
  if (enriched.opponentActive) {
    enriched.opponentActive = enriched.opponentActive.map(p => 
      p ? DataAdapter.inferUnknowns(p) : null
    );
  }
  
  // Enrich opponent bench
  if (enriched.opponentBench) {
    enriched.opponentBench = enriched.opponentBench.map(p =>
      p ? DataAdapter.inferUnknowns(p) : null
    );
  }
  
  return enriched;
}

/**
 * Enumerate all possible actions for a Pokemon
 */
function enumerateActions(pokemon, slotIndex, state) {
  const actions = [];
  const MoveData = getMoveData();
  
  // Get available moves
  const moves = pokemon.availableMoves || pokemon.moves || pokemon.likelyMoves || [];
  
  for (const moveName of moves) {
    const moveData = MoveData.getMove(moveName);
    
    if (MoveData.isSpreadMove(moveName)) {
      // Spread move - no specific target
      actions.push({
        type: 'move',
        move: moveName,
        target: null,
        isSpread: true
      });
    } else if (moveData?.target === 'self' || moveData?.target === 'field') {
      // Self-targeting or field move
      actions.push({
        type: 'move',
        move: moveName,
        target: 'self'
      });
    } else if (moveData?.target === 'ally') {
      // Ally targeting
      actions.push({
        type: 'move',
        move: moveName,
        target: 'ally'
      });
    } else {
      // Single target - can target each opponent
      const opponents = (state.opponentActive || []).filter(p => p && p.hp > 0);
      for (const opp of opponents) {
        actions.push({
          type: 'move',
          move: moveName,
          target: opp.name
        });
      }
      
      // Special handling for moves that can/should target allies too
      // (Even if Showdown explicitly says 'normal', we want to consider self-targeting strategies)
      const allyTargetableMoves = [
          'psych up', 'coaching', 'pollen puff', 'decorate', 
          'aromatic mist', 'floral healing', 'pain split', 'entrainment', 'simple beam'
      ];
      
      if (allyTargetableMoves.includes(moveName.toLowerCase())) {
          const partners = (state.playerActive || []).filter(p => p && p.hp > 0 && p.name !== pokemon.name);
          for (const partner of partners) {
               actions.push({
                  type: 'move',
                  move: moveName,
                  target: partner.name
               });
          }
      }
    }
  }
  
  // Add switch options
  const bench = state.playerBench || [];
  for (const benchMon of bench) {
    if (benchMon && benchMon.hp > 0) {
      actions.push({
        type: 'switch',
        switchTo: benchMon.name
      });
    }
  }
  
  // Add Tera option if available and unused
  if (state.canTera && !pokemon.teraActive) {
    // Get likely best Tera type
    const DataAdapter = getDataAdapter();
    const likelyTera = DataAdapter.getMostLikelyTera(pokemon.name);
    if (likelyTera) {
      actions.push({
        type: 'tera',
        teraType: likelyTera.tera
      });
    }
  }
  
  return actions;
}

// =============================================================================
// SCORING
// =============================================================================

// =============================================================================
// SCORING
// =============================================================================

/**
 * Score an action using the multiplicative formula
 * REMOVED: Rationale text generation
 */
function scoreAction(action, pokemon, state) {
  // Base tactical score
  const tacticalScore = scoreTactical(action, pokemon, state);
  
  // Role alignment score
  const roleScore = scoreRole(action, pokemon, state);
  
  // Blindspot bonus
  const blindspotScore = scoreBlindspot(action, pokemon, state);
  
  // Meta consideration
  const metaScore = scoreMeta(action, pokemon, state);

  // VGC Context specific rules (Turn 1, etc)
  const contextScore = scoreContext(action, pokemon, state);
  
  // Final score = Tactical × Role × Blindspot × Meta × Context
  const finalScore = tacticalScore * roleScore * blindspotScore * metaScore * contextScore;
  
  return {
    ...action,
    score: Math.round(finalScore * 100) / 100,
    // No breakdown or rationale needed for UI anymore
  };
}

/**
 * Score tactical value of an action
 */
function scoreTactical(action, pokemon, state) {
  if (action.type === 'move') {
    return scoreMoveAction(action, pokemon, state);
  } else if (action.type === 'switch') {
    return scoreSwitchAction(action, pokemon, state);
  } else if (action.type === 'tera') {
    return scoreTeraAction(action, pokemon, state);
  }
  return 1.0;
}

/**
 * Score a move action
 */
function scoreMoveAction(action, pokemon, state) {
  const MoveData = getMoveData();
  const TypeChart = getTypeChart();
  const DataAdapter = getDataAdapter();
  
  const moveData = MoveData.getMove(action.move);
  let score = 1.0;
  
  // === CONDITIONAL VALIDITY CHECKS ===
  const moveName = action.move.toLowerCase();
  
  // Sleep Talk / Snore require sleep
  if ((moveName === 'sleep talk' || moveName === 'snore') && pokemon.status !== 'slp') {
      return 0;
  }
  
  // Dream Eater / Nightmare require target sleep
  const target = (state.opponentActive || []).find(p => p?.name === action.target);
  if (target && (moveName === 'dream eater' || moveName === 'nightmare') && target.status !== 'slp') {
      return 0;
  }
  
  // Baton Pass requires boosts to be useful (per user request)
  if (moveName === 'baton pass') {
     const hasBoosts = pokemon.boosts && Object.values(pokemon.boosts).some(val => val > 0);
     if (!hasBoosts) return 0;
  }
  
  // Fling / Natural Gift require item
  if ((moveName === 'fling' || moveName === 'natural gift') && !pokemon.item && !pokemon.revealedItem) {
      return 0;
  }
  
  // Stored Power / Power Trip require boosts
  if ((moveName === 'stored power' || moveName === 'power trip')) {
      const hasBoosts = pokemon.boosts && Object.values(pokemon.boosts).some(val => val > 0);
      if (!hasBoosts) return 0; 
  }
  
  // Psych Up / Costar require target with boosts
  if (moveName === 'psych up' || moveName === 'costar') {
      const target = (state.opponentActive || []).find(p => p?.name === action.target) || 
                     (state.playerActive || []).find(p => p?.name === action.target);
      
      const hasBoosts = target && target.boosts && Object.values(target.boosts).some(val => val > 0);
      if (!hasBoosts) return 0;
  }

  // === WEATHER / TERRAIN DEPENDENCIES ===
  
  // Aurora Veil requires Snow
  if (moveName === 'aurora veil' && state.field.weather !== 'snow') {
      return 0; // Fails
  }
  
  // Steel Roller requires Terrain
  if (moveName === 'steel roller' && !state.field.terrain) {
      return 0; // Fails
  }
  
  // Solar Beam / Solar Blade require Sun OR Power Herb
  if ((moveName === 'solar beam' || moveName === 'solar blade') && state.field.weather !== 'sun') {
      const hasPowerHerb = pokemon.item === 'Power Herb' || pokemon.revealedItem === 'Power Herb';
      if (!hasPowerHerb) score *= 0.1; // Takes 2 turns, usually bad in VGC without sun
  }
  
  // Electro Shot requires Rain OR Power Herb
  if (moveName === 'electro shot' && state.field.weather !== 'rain') {
      const hasPowerHerb = pokemon.item === 'Power Herb' || pokemon.revealedItem === 'Power Herb';
      if (!hasPowerHerb) score *= 0.1;
  }
  
  // Meteor Beam requires Power Herb (almost exclusively)
  if (moveName === 'meteor beam') {
       const hasPowerHerb = pokemon.item === 'Power Herb' || pokemon.revealedItem === 'Power Herb';
       if (!hasPowerHerb) score *= 0.2;
  }
  
  // Expanding Force - Check Terrain for target implications? 
  // (Handled by spread logic but we can boost it)
  if (moveName === 'expanding force' && state.field.terrain === 'psychic') {
      score *= 1.2;
  }
  
  // Protect moves
  if (MoveData.isProtectMove(action.move)) {
    return scoreProtect(action, pokemon, state);
  }
  
  // Status moves
  if (moveData?.category === 'status') {
    return scoreStatusMove(action, pokemon, state);
  }
  
  // Spread moves
  if (action.isSpread) {
    return scoreSpreadMove(action, pokemon, state);
  }
  
  // Single target attack
  if (!target) return 0.5;
  
  const targetTypes = DataAdapter.getTypes(target.name) || target.types || [];
  
  // Safety check: If types are unknown, penalty to avoid confident bad advice
  if (!targetTypes || targetTypes.length === 0) {
    score *= 0.2;
  }
  
  const moveType = moveData?.type;
  
  // === IMMUNITIES & ABILITIES ===
  
  // Powder moves vs Grass types / Safety Goggles / Overcoat
  if (moveData?.flags?.powder) {
      if (targetTypes.includes('grass')) return 0;
      if (target.item === 'Safety Goggles' || target.revealedItem === 'Safety Goggles') return 0;
      if (target.ability === 'Overcoat' || target.revealedAbility === 'Overcoat') return 0;
  }
  
  // Prankster vs Dark types
  const myAbility = pokemon.ability || pokemon.likelyAbility;
  if (myAbility === 'Prankster' && moveData?.category === 'status') {
      if (targetTypes.includes('dark')) return 0;
  }
  
  // Good as Gold (Gholdengo) blocks Status moves
  const targetAbility = target.ability || target.revealedAbility || target.likelyAbility;
  if (targetAbility === 'Good as Gold' && moveData?.category === 'status' && action.target !== 'self' && action.target !== 'ally') {
      return 0;
  }

  // Type effectiveness
  const effectiveness = TypeChart.getEffectivenessVsDual(moveType, targetTypes);
  
  // STRICT IMMUNITY CHECK
  if (effectiveness === 0) {
    return 0; // Don't suggest moves that do nothing
  }
  
  // Sucker Punch / Thunderclap / Upper Hand Logic
  // These moves fail if the opponent doesn't attack
  if (moveName === 'sucker punch' || moveName === 'thunderclap' || moveName === 'upper hand') {
       // Heuristic: If target is Support or Setup, they are LESS likely to attack directly
       const RoleSystem = getRoleSystem();
       const targetRoles = RoleSystem.detectRoles(target, state);
       
       if (targetRoles.includes('SUPPORT') || targetRoles.includes('TRICK_ROOM_SETTER') || targetRoles.includes('SETUP_SWEEPER')) {
           score *= 0.4; // Risky click
       }
       
       // Upper Hand fails if target doesn't use Priority
       if (moveName === 'upper hand') {
           score *= 0.5; // Very situational, assume risky unless we know more
       }
  }

  if (effectiveness >= 2) {
    score *= 1.4;
  } else if (effectiveness < 1) {
    score *= 0.5; // Heavier penalty for resisted moves
  }
  
  // STAB
  const pokemonTypes = DataAdapter.getTypes(pokemon.name) || pokemon.types || [];
  const stabMult = TypeChart.getSTABMultiplier(moveType, pokemonTypes, pokemon.teraType, pokemon.teraActive);
  if (stabMult > 1) {
    score *= 1.1;
  }
  
  // Base power (using updated moveData if available)
  const power = moveData?.basePower || 0;
  if (power >= 100) {
    score *= 1.15;
  } else if (power === 0) {
     // Variable power moves (Grass Knot, Low Kick, Seismic Toss, Night Shade, etc)
     // Give them a generic passable score if type effective
     score *= 1.0; 
  }
  
  // Priority
  if (moveData?.priority > 0) {
    score *= 1.1;
  }
  
  // KO potential
  if (target.hp <= 50 && effectiveness >= 1) {
    score *= 1.3;
  }
  
  // Win condition targeting
  const RoleSystem = getRoleSystem();
  if (RoleSystem.isWinCondition(target.name)) {
    score *= 1.25;
  }
  
  return score;
}

/**
 * Score a protect action
 */
function scoreProtect(action, pokemon, state) {
  let score = 0.8; // Baseline - protect is often suboptimal
  const BlindspotWeights = getBlindspotWeights();
  
  // Check if both opponents might target us
  const opponentActive = state.opponentActive || [];
  const threatCount = opponentActive.filter(opp => {
    if (!opp || opp.hp <= 0) return false;
    return couldTarget(opp, pokemon, state);
  }).length;
  
  if (threatCount >= 2) {
    score *= BlindspotWeights.getWeight('PROTECT_DOUBLE_TARGETED');
  }
  
  // Protect to stall speed control
  if (state.field?.tailwind && state.field.tailwindTurns <= 1) {
    score *= 1.2;
  }
  
  if (state.field?.trickRoom && state.field.trickRoomTurns <= 1) {
    score *= 1.2;
  }
  
  // Penalty if used last turn
  if (pokemon.lastMove && pokemon.lastMove.toLowerCase().includes('protect')) {
    score *= 0.05; // Almost never suggest double protect (5% chance in VGC)
  }
  
  // SURVIVAL PRIORITY
  if (isUnderKoThreat(pokemon, state)) {
      score *= 1.5;
  }
  
  return score;
}

/**
 * Score a status move
 */
function scoreStatusMove(action, pokemon, state) {
  const MoveData = getMoveData();
  const BlindspotWeights = getBlindspotWeights();
  
  let score = 0.9;
  const moveName = action.move.toLowerCase();
  
  // Speed control
  if (MoveData.isSpeedControl(action.move)) {
    if (moveName === 'tailwind' && state.turn <= 1) {
      score *= BlindspotWeights.getWeight('TAILWIND_TURN_ONE');
    } else if (moveName === 'trick room') {
      score *= BlindspotWeights.getWeight('TRICK_ROOM_VS_FAST');
    } else {
      score *= BlindspotWeights.getWeight('SPREAD_SPEED_DROP');
    }
  }
  
  // Fake Out
  if (moveName === 'fake out') {
      if (pokemon.justSwitchedIn !== false && state.turn === 1) {
          score *= 2.0; // Huge priority Turn 1
      } else if (pokemon.justSwitchedIn !== false) {
           const target = (state.opponentActive || []).find(p => p?.name === action.target);
           score *= BlindspotWeights.getWeight('FAKE_OUT_DENY');
      } else {
          score *= 0; // Cannot use Fake Out if not switched in
      }
  }
  
  // Helping Hand
  if (moveName === 'helping hand') {
    score *= 1.1;
  }
  
  // Taunt / Encore / Haze / Roar / Whirlwind (Disruption)
  const disruptionMoves = ['taunt', 'encore', 'haze', 'roar', 'whirlwind', 'clear smog'];
  if (disruptionMoves.includes(moveName)) {
      if (isOpponentSetupThreat(state)) {
          score *= 1.4; // Anti-setup priority
      } else if (moveName === 'taunt') {
          score *= BlindspotWeights.getWeight('TAUNT_SUPPORT');
      }
  }
  
  // Redirection
  if (MoveData.isRedirectMove(action.move)) {
    // Standard redirection value
    score *= BlindspotWeights.getWeight('REDIRECT_PROTECT_PARTNER');
    
    // SURVIVAL PRIORITY FOR PARTNER
    if (isPartnerUnderKoThreat(pokemon, state)) {
        score *= 1.5;
    }
  }
  
  return score;
}

/**
 * Score a spread move
 */
function scoreSpreadMove(action, pokemon, state) {
  const TypeChart = getTypeChart();
  const MoveData = getMoveData();
  const DataAdapter = getDataAdapter();
  const BlindspotWeights = getBlindspotWeights();
  
  const moveData = MoveData.getMove(action.move);
  let score = 1.0;
  
  const opponentActive = (state.opponentActive || []).filter(p => p && p.hp > 0);
  if (opponentActive.length === 0) return 0.5;
  
  // Check effectiveness against each target
  let totalEff = 0;
  let superEffCount = 0;
  let immuneCount = 0;
  
  for (const opp of opponentActive) {
    const types = DataAdapter.getTypes(opp.name) || opp.types || [];
    const eff = TypeChart.getEffectivenessVsDual(moveData?.type, types);
    totalEff += eff;
    if (eff >= 2) superEffCount++;
    if (eff === 0) immuneCount++;
  }
  
  if (immuneCount > 0) {
    score *= 0.7;
  }
  
  if (superEffCount >= 2) {
    score *= 1.4;
  } else if (superEffCount === 1) {
    score *= 1.15;
  }
  
  // Speed drop spread moves
  if (moveData?.effect?.speedDrop) {
    score *= BlindspotWeights.getWeight('SPREAD_SPEED_DROP');
  }
  
  // Both damaged
  const bothDamaged = opponentActive.every(p => p.hp < 75);
  if (bothDamaged) {
    score *= BlindspotWeights.getWeight('SPREAD_BOTH_DAMAGED');
  }
  
  return score;
}

/**
 * Score a switch action
 */
function scoreSwitchAction(action, pokemon, state) {
  const DataAdapter = getDataAdapter();
  const TypeChart = getTypeChart();
  const BlindspotWeights = getBlindspotWeights();
  const RoleSystem = getRoleSystem();
  
  // Boosted base score per user request
  let score = 0.85; 
  
  const switchTo = state.playerBench?.find(p => p?.name === action.switchTo);
  if (!switchTo) return 0.4;
  
  const currentTypes = DataAdapter.getTypes(pokemon.name) || pokemon.types || [];
  const switchTypes = DataAdapter.getTypes(switchTo.name) || switchTo.types || [];
  
  // Check against all active opponents
  const opponentActive = state.opponentActive || [];
  let defensiveValue = 0;
  let offensiveValue = 0;
  
  for (const opp of opponentActive) {
    if (!opp || opp.hp <= 0) continue;
    
    // DEFENSIVE UTILITY
    const oppTypes = DataAdapter.getTypes(opp.name) || opp.types || [];
    const superEffTypes = TypeChart.getSuperEffectiveTypes(currentTypes);
    
    // Would switching in resist what opponents could throw?
    for (const seType of superEffTypes) {
      const currentEff = TypeChart.getEffectivenessVsDual(seType, currentTypes);
      const switchEff = TypeChart.getEffectivenessVsDual(seType, switchTypes);
      
      if (switchEff < currentEff) {
        // We are improving our defensive position
        defensiveValue += 1;
        break; // Count once per opponent per threat type
      }
    }
    
    // OFFENSIVE PIVOT UTILITY (Aggressive Switching)
    // Does the switch-in have immediate offensive pressure?
    const switchInStab = switchTypes; // Simplified, assuming STAB is primary offense
    for (const myType of switchInStab) {
        const eff = TypeChart.getEffectivenessVsDual(myType, oppTypes);
        if (eff >= 2) {
            offensiveValue += 1;
        }
    }
  }
  
  if (defensiveValue > 0) {
      score *= BlindspotWeights.getWeight('SWITCH_BETTER_MATCHUP');
  }
  
  if (offensiveValue > 0) {
      score *= 1.25; // Good offensive switch
  }
  
  // Preserve win condition
  if (RoleSystem.isWinCondition(pokemon.name) && pokemon.hp < 50) {
    score *= BlindspotWeights.getWeight('SWITCH_PRESERVE_WIN_CON');
  }
  
  // Switch into Intimidate
  const switchAbility = switchTo.ability?.toLowerCase() || 
    DataAdapter.getMostLikelyAbility(switchTo.name)?.ability?.toLowerCase();
  if (switchAbility === 'intimidate') {
    score *= BlindspotWeights.getWeight('SWITCH_INTO_INTIMIDATE');
  }
  
  // Burned physical attacker
  if (pokemon.status === 'brn' && RoleSystem.detectRoles(pokemon, state).includes('PHYSICAL_ATTACKER')) {
    score *= BlindspotWeights.getWeight('SWITCH_OUT_BURNED_PHYSICAL');
  }
  
  // SURVIVAL PRIORITY
  // Only boost switch if we are under threat AND the switch target has decent HP (>50%)
  if (isUnderKoThreat(pokemon, state) && switchTo.hpPercent > 50) {
      score *= 1.5;
  }
  
  return score;
}

/**
 * Score a Tera action
 */
function scoreTeraAction(action, pokemon, state) {
  const BlindspotWeights = getBlindspotWeights();
  const DataAdapter = getDataAdapter();
  const TypeChart = getTypeChart();
  
  let score = 0.95; // Tera is a limited resource, but powerful
  
  const teraType = action.teraType;
  if (!teraType) return 0.5; // Should not happen
  
  // === DEFENSIVE UTILITY ===
  // Does this Tera type resist the opponent?
  const opponentActive = state.opponentActive || [];
  let defensiveWin = false;
  
  for (const opp of opponentActive) {
      if (!opp || opp.hp <= 0) continue;
      const oppTypes = DataAdapter.getTypes(opp.name) || opp.types || [];
      
      // Heuristic: Check if opponent has STAB moves that threaten us
      for (const t of oppTypes) {
           const eff = TypeChart.getEffectivenessVsSingle(t, teraType);
           if (eff < 1) { 
               // We resist their STAB
               defensiveWin = true;
           }
      }
  }
  
  if (defensiveWin) {
       score *= 1.25;
  }
  
  // === OFFENSIVE UTILITY ===
  // Does this Tera type give us a Super Effective STAB bonus?
  // (i.e. we are Tera-ing into a type to kill something)
  let offensiveWin = false;
  const myMoves = pokemon.availableMoves || pokemon.moves || [];
  
  for (const opp of opponentActive) {
      if (!opp || opp.hp <= 0) continue;
      const oppTypes = DataAdapter.getTypes(opp.name) || opp.types || [];
      
      // Check our moves
      for (const m of myMoves) {
          // If move matches Tera Type...
           // get move data... we need MoveData
           const MoveData = getMoveData(); 
           const mData = MoveData.getMove(m);
           if (mData && mData.type && mData.type.toLowerCase() === teraType.toLowerCase()) {
                // It is a STAB move now. Is it super effective?
                const eff = TypeChart.getEffectivenessVsDual(mData.type, oppTypes);
                if (eff >= 2) {
                    offensiveWin = true;
                }
           }
      }
  }
  
  if (offensiveWin) {
      score *= 1.4; // Big boost for offensive pressure
  }
  
  // Check if Tera would help survive (Emergency)
  if (pokemon.hp < 40 && defensiveWin) {
    score *= BlindspotWeights.getWeight('TERA_SURVIVE_KO');
  }
  
  // Penalize if no clear purpose
  if (!offensiveWin && !defensiveWin && state.turn <= 2) {
    score *= BlindspotWeights.getWeight('TERA_WASTEFUL');
  }
  
  return score;
}

/**
 * Score VGC Context (Turn 1, etc)
 */
function scoreContext(action, pokemon, state) {
    let score = 1.0;
    
    // EARLY GAME RULES (Turn 1 & 2)
    if (state.turn <= 2) {
        if (action.type === 'move') {
            const moveName = action.move.toLowerCase();
            
            // BOOST SETUP MOVES
            // Common VGC setup moves
            const setupMoves = [
                'swords dance', 'dragon dance', 'nasty plot', 'calm mind', 
                'quiver dance', 'shell smash', 'bulk up', 'coil', 
                'shift gear', 'tail glow', 'iron defense', 'amnesia', 'cosmic power'
            ];
            
            if (setupMoves.includes(moveName)) {
                score *= 1.5;
            }
            
            // Turn 1 Specifics
            if (state.turn === 1) {
                const turn1Boosts = ['fake out', 'protect', 'tailwind', 'trick room', 'reflect', 'light screen', 'aurora veil'];
                if (turn1Boosts.includes(moveName)) {
                    score *= 1.5;
                }
            }
        }
        
        // BAN SWITCHING ON TURN 1 only
        if (state.turn === 1 && action.type === 'switch') {
            return 0.0;
        }
    }
    
    return score;
}

/**
 * Score role alignment
 */
function scoreRole(action, pokemon, state) {
  const RoleSystem = getRoleSystem();
  const MoveData = getMoveData();
  const DataAdapter = getDataAdapter();
  
  let score = 1.0;
  const roles = RoleSystem.detectRoles(pokemon, state);
  const moveData = action.type === 'move' ? MoveData.getMove(action.move) : null;
  
  // === SUPPORT ROLE LOGIC ===
  if (roles.includes('SUPPORT')) {
      if (action.type === 'move') {
          // If attacking move, penalize heavily
          if (moveData && moveData.category !== 'status') {
               // Only penalize if we actually HAVE support moves available
               // (If we are Taunted or only have weak attacks, we might have to attack)
               const hasSupport = (pokemon.availableMoves || []).some(m => {
                   const d = MoveData.getMove(typeof m === 'string' ? m : m.name);
                   return d && d.category === 'status';
               });
               
               if (hasSupport) {
                   score *= 0.2; // Don't attack if you are support
               }
          } else {
              // Status moves boosted
              score *= 1.2;
          }
      }
  }
  
  // === SWEEPER LOGIC ===
  if (roles.includes('PHYSICAL_ATTACKER') || roles.includes('SPECIAL_ATTACKER') || roles.includes('SETUP_SWEEPER')) {
      // Don't switch out if you are the carry, unless into a safe pivot/tank
      if (action.type === 'switch') {
          const switchTo = state.playerBench?.find(p => p?.name === action.switchTo);
          if (switchTo) {
               // Infer role of switch target
               // We need a way to detect role of bench mon... 
               // Reuse detectRoles but ensure it handles minimal data
               const switchRoles = RoleSystem.detectRoles(switchTo, state);
               
               if (switchRoles.includes('TANK') || switchRoles.includes('WALL') || switchRoles.includes('PIVOT') || switchRoles.includes('INTIMIDATE_PIVOT')) {
                   score *= 1.2; // Good pivot
               } else {
                   score *= 0.3; // Don't switch a sweeper into another squishy
               }
          }
      }
  }

  if (action.type === 'move') {
    // Speed control role wants to set speed control
    if (roles.includes('SPEED_CONTROL') && MoveData.isSpeedControl(action.move)) {
      score *= 1.15;
    }
    
    // Trick Room setter using Trick Room
    if (roles.includes('TRICK_ROOM_SETTER') && action.move?.toLowerCase() === 'trick room') {
      score *= 1.2;
    }
  }
  
  return score;
}

/**
 * Score blindspot adjustments
 */
function scoreBlindspot(action, pokemon, state) {
  // Most blindspot logic is already applied in tactical scoring
  // This is for any remaining adjustments
  return 1.0;
}

/**
 * Score meta considerations
 */
function scoreMeta(action, pokemon, state) {
  const DataAdapter = getDataAdapter();
  let score = 1.0;
  
  // HIGH USAGE BIAS
  if (action.type === 'move') {
      const usage = DataAdapter.getUsageData(pokemon.name);
      if (usage && usage.moves) {
          // Normalizes move name
          const normalizedMove = action.move.toLowerCase().replace(/[^a-z0-9]/g, '');
          let moveProb = 0;
          
          for (const [m, prob] of Object.entries(usage.moves)) {
              if (m.toLowerCase().replace(/[^a-z0-9]/g, '') === normalizedMove) {
                  moveProb = prob;
                  break;
              }
          }
          
          if (moveProb > 0.60) {
              score *= 1.25; // Very common standard move
          } else if (moveProb > 0.30) {
              score *= 1.1; // Common move
          }
      }
  }
  
  return score;
}

// =============================================================================
// UTILITIES
// =============================================================================

/**
 * Check if the current Pokemon is under immediate KO threat
 */
function isUnderKoThreat(pokemon, state) {
    if (!pokemon) return false;
    
    // 1. HP Threshold: < 35% is always dangerous
    if (pokemon.hp < 35) return true;
    
    // 2. Type Disadvantage Threshold: < 60% and facing threats
    if (pokemon.hp < 60) {
        const TypeChart = getTypeChart();
        const DataAdapter = getDataAdapter();
        const myTypes = DataAdapter.getTypes(pokemon.name) || pokemon.types || [];
        
        const opponentActive = state.opponentActive || [];
        for (const opp of opponentActive) {
            if (!opp || opp.hp <= 0) continue;
            
            // Check if opponent has super effective STAB or known moves
            const oppTypes = DataAdapter.getTypes(opp.name) || opp.types || [];
            
            // Assume opponent has STAB moves
            for (const type of oppTypes) {
                if (TypeChart.getEffectivenessVsDual(type, myTypes) >= 2) {
                    return true;
                }
            }
        }
    }
    
    return false;
}

/**
 * Check if the PARTNER of the current Pokemon is under KO threat
 */
function isPartnerUnderKoThreat(pokemon, state) {
    // Find my partner (the other active player pokemon)
    const partner = (state.playerActive || []).find(p => p && p.name !== pokemon.name && p.hp > 0);
    if (!partner) return false;
    
    return isUnderKoThreat(partner, state);
}

/**
 * Check if any Active Opponent is a Setup threat
 */
function isOpponentSetupThreat(state) {
    const RoleSystem = getRoleSystem();
    const DataAdapter = getDataAdapter();
    const opponents = state.opponentActive || [];
    
    for (const opp of opponents) {
        if (!opp || opp.hp <= 0) continue;
        
        // We might need to infer roles if moves are unknown
        let roles = [];
        if (opp.moves && opp.moves.length > 0) {
             roles = RoleSystem.detectRoles(opp, state);
        } else {
             // Use inferred data from stats
             const inferred = DataAdapter.inferUnknowns(opp);
             roles = RoleSystem.detectRoles(inferred, state);
        }
        
        if (roles.includes('SETUP_SWEEPER') || roles.includes('TRICK_ROOM_SETTER')) {
            return true;
        }
    }
    return false;
}

/**
 * Check if an opponent could target a specific Pokemon
 */
function couldTarget(opponent, pokemon, state) {
  // Simple heuristic - assume they could target
  return true;
}

/**
 * Calculate speed order for the field
 */
function calculateSpeedOrder(gameState) {
  const DataAdapter = getDataAdapter();
  const all = [];
  
  // Player active
  for (const p of (gameState.playerActive || [])) {
    if (!p || p.hp <= 0) continue;
    const stats = DataAdapter.getBaseStats(p.name);
    all.push({
      name: p.name,
      side: 'player',
      speed: stats?.spe || 100,
      item: p.item,
      ability: p.ability
    });
  }
  
  // Opponent active
  for (const p of (gameState.opponentActive || [])) {
    if (!p || p.hp <= 0) continue;
    const stats = DataAdapter.getBaseStats(p.name);
    all.push({
      name: p.name,
      side: 'opponent',
      speed: stats?.spe || 100,
      item: p.item,
      ability: p.ability
    });
  }
  
  // Sort by speed (considering Trick Room)
  const inTrickRoom = gameState.field?.trickRoom;
  all.sort((a, b) => inTrickRoom ? a.speed - b.speed : b.speed - a.speed);
  
  return all;
}

// =============================================================================
// EXPORTS
// =============================================================================

window.BattleCoach = {
  initialize,
  isReady,
  analyze,
  analyzeSlot,
  calculateSpeedOrder
};
