/**
 * Battle Coach - Data Adapter
 * Provides unified interface to extension's compact JSON data files
 */

// Data stores - populated on init
let POKEMON_STATS = null;
let SMOGON_STATS = null;

/**
 * Initialize with loaded JSON data
 */
function initializeData({ pokemonStats, smogonStats }) {
  POKEMON_STATS = pokemonStats;
  SMOGON_STATS = smogonStats;
  console.log('[DataAdapter] Initialized with', 
    Object.keys(pokemonStats || {}).length, 'pokemon,',
    Object.keys(smogonStats || {}).length, 'usage entries');
}

/**
 * Normalize species name for lookups
 */
function normalizeName(name) {
  if (!name) return '';
  return name.toLowerCase()
    .replace(/['']/g, '')  // Remove apostrophes
    .replace(/-/g, '')     // Remove hyphens
    .replace(/\s+/g, '');  // Remove spaces
}

/**
 * Try multiple name formats for lookup
 */
function tryLookup(db, name) {
  if (!db || !name) return null;
  
  const normalized = normalizeName(name);
  
  // Try exact match first
  if (db[name.toLowerCase()]) return db[name.toLowerCase()];
  if (db[normalized]) return db[normalized];
  
  // Try with hyphen preserved
  const hyphenated = name.toLowerCase().replace(/\s+/g, '-');
  if (db[hyphenated]) return db[hyphenated];
  
  // Try lowercase with spaces removed
  const lowerNoSpace = name.toLowerCase().replace(/\s+/g, '');
  if (db[lowerNoSpace]) return db[lowerNoSpace];
  
  return null;
}

/**
 * Get base stats for a species
 */
function getBaseStats(name) {
  const data = tryLookup(POKEMON_STATS, name);
  return data?.baseStats || null;
}

/**
 * Get types for a species
 */
function getTypes(name) {
  const data = tryLookup(POKEMON_STATS, name);
  if (!data?.types) return null;
  // Normalize to lowercase for consistency
  return data.types.map(t => t.toLowerCase());
}

/**
 * Get usage data for a species
 */
function getUsageData(name) {
  return tryLookup(SMOGON_STATS, name) || null;
}

/**
 * Get most likely item for a species
 */
function getMostLikelyItem(name) {
  const usage = getUsageData(name);
  if (!usage?.items) return null;
  
  const entries = Object.entries(usage.items);
  if (entries.length === 0) return null;
  
  entries.sort((a, b) => b[1] - a[1]);
  return { item: entries[0][0], probability: entries[0][1] };
}

/**
 * Get most likely ability for a species
 */
function getMostLikelyAbility(name) {
  const usage = getUsageData(name);
  if (!usage?.abilities) return null;
  
  const entries = Object.entries(usage.abilities);
  if (entries.length === 0) return null;
  
  entries.sort((a, b) => b[1] - a[1]);
  return { ability: entries[0][0], probability: entries[0][1] };
}

/**
 * Get most likely moves for a species
 */
function getMostLikelyMoves(name, count = 4) {
  const usage = getUsageData(name);
  if (!usage?.moves) return [];
  
  return Object.entries(usage.moves)
    .map(([move, prob]) => ({ move, probability: prob }))
    .sort((a, b) => b.probability - a.probability)
    .slice(0, count);
}

/**
 * Get most likely tera type for a species
 */
function getMostLikelyTera(name) {
  const usage = getUsageData(name);
  if (!usage?.tera_types) return null;
  
  const entries = Object.entries(usage.tera_types);
  if (entries.length === 0) return null;
  
  entries.sort((a, b) => b[1] - a[1]);
  return { tera: entries[0][0], probability: entries[0][1] };
}

/**
 * Infer unknown data for a Pokemon based on usage stats
 */
function inferUnknowns(pokemon) {
  const name = pokemon.name;
  const enriched = { ...pokemon };
  
  // Infer types if not known
  if (!enriched.types || enriched.types.length === 0) {
    enriched.types = getTypes(name) || [];
  }
  
  // Infer ability if not known
  if (!enriched.ability) {
    const likelyAbility = getMostLikelyAbility(name);
    if (likelyAbility) {
      enriched.likelyAbility = likelyAbility.ability;
    }
  }
  
  // Infer item if not known
  if (!enriched.item) {
    const likelyItem = getMostLikelyItem(name);
    if (likelyItem) {
      enriched.likelyItem = likelyItem.item;
    }
  }
  
  // Infer moves
  if (!enriched.moves || enriched.moves.length === 0) {
    const likelyMoves = getMostLikelyMoves(name, 4);
    enriched.likelyMoves = likelyMoves.map(m => m.move);
  }
  
  // Infer tera type
  if (!enriched.teraType) {
    const likelyTera = getMostLikelyTera(name);
    if (likelyTera) {
      enriched.likelyTera = likelyTera.tera;
    }
  }
  
  return enriched;
}

/**
 * Get all usage statistics (for UI display)
 */
function getFullUsageStats(name) {
  return getUsageData(name);
}

/**
 * Check if data is initialized
 */
function isInitialized() {
  return POKEMON_STATS !== null && SMOGON_STATS !== null;
}

// Expose to global scope
window.DataAdapter = {
  initializeData,
  isInitialized,
  getBaseStats,
  getTypes,
  getUsageData,
  getMostLikelyItem,
  getMostLikelyAbility,
  getMostLikelyMoves,
  getMostLikelyTera,
  inferUnknowns,
  getFullUsageStats
};
