/**
 * Battle Coach - State Reader
 * Parses Pokémon Showdown battle DOM to extract structured game state
 * Works in both player and spectator modes
 */

class ShowdownStateReader {
  constructor() {
    this.battleRoom = null;
    // Cache for used moves from battle log
    this.usedMoves = {}; // { normalizedName: Set of moves }
    this.benchHistory = {}; // { normalizedSpecies: { species, name, lastHp, lastStatus, usedMoves } }
    this.nicknameMap = {}; // { normalizedNickname: speciesKey }
    this.revealedData = {}; // { speciesKey: { ability: string, item: string } }
    this.lastPlayerSideClass = null; // Unused - kept for compatibility
    this.lastTurn = 0;
    
    // STABILITY FIX: Cache the team state to avoid reading during animations
    // STABILITY FIX: Cache the team state to avoid reading during animations
    this.cachedTeamState = null;
  }

  /**
   * Clear all internal state
   */
  clearState() {
    this.battleRoom = null;
    this.usedMoves = {};
    this.benchHistory = {};
    this.nicknameMap = {};
    this.revealedData = {};
    this.boosts = {};
    this.lastPlayerSideClass = null;
    this.lastTurn = 0;
    this.cachedTeamState = null;
    console.log('🥋 StateReader cleared');
  }

  findBattleRoom() {
    console.log('🥋 Looking for battle room...');
    const selectors = ['.battle', '.battle-log', '.innerbattle'];
    for (const selector of selectors) {
      const elements = document.querySelectorAll(selector);
      console.log(`🥋 Selector "${selector}" found ${elements.length} elements`);
      for (const element of elements) {
        if (element.offsetParent !== null) {
          const room = element.closest('.ps-room') || element;
          console.log('🥋 Found visible battle room');
          return room;
        }
      }
    }
    const fallback = document.querySelector('.battle');
    if (fallback) {
      console.log('🥋 Using fallback battle element');
      return fallback.closest('.ps-room') || fallback;
    }
    console.log('🥋 No battle room found!');
    return null;
  }

  readState() {
    this.battleRoom = this.findBattleRoom();
    if (!this.battleRoom) {
      // No battle active - clear state
      if (this.cachedTeamState) {
          this.clearState();
      }
      return null;
    }

    this.parseBattleLog();

    // STABILITY CHECK: Are controls visible?
    const startStable = performance.now();
    const controlsVisible = !!document.querySelector('.battle-controls .whatdo') || 
                           !!document.querySelector('.movemenu') || 
                           !!document.querySelector('.switchmenu');
    
    // Always read basic metadata
    // Reset tracking if we detect a new battle (turn 1 after a higher turn)
    const currentTurn = this.readTurn();
    if (currentTurn === 1 && this.lastTurn > 1) {
      this.usedMoves = {};
      this.benchHistory = {};
      this.nicknameMap = {};
      this.revealedData = {};
      this.boosts = {};
      this.lastPlayerSideClass = null; 
      this.cachedTeamState = null; // Clear cache on new battle
    }
    this.lastTurn = currentTurn;

    // Get active prompt name for move assignment
    let activePromptName = null;
    const promptElement = document.querySelector('.battle-controls .whatdo');
    if (promptElement) {
      const text = promptElement.innerText || '';
      const match = text.match(/What will (.+?) do/i);
      if (match) {
        activePromptName = match[1].trim();
      }
    }
    
    // LOGIC:
    // If controls are visible -> WE ARE IN INPUT PHASE. Read everything fresh and update cache.
    // If controls hidden -> WE ARE IN ANIMATION/WAIT PHASE. Use cache for teams, but update field/log data.
    
    let playerPokemon = [];
    let opponentPokemon = [];
    let playerBench = [];
    let opponentBench = [];
    
    // Force full parse if we have no cache yet, or if it is the input phase
    const shouldFullParse = controlsVisible || !this.cachedTeamState;

    if (shouldFullParse) {
      // --- FULL DOM READ ---
      console.log('🥋 Input Phase (or no cache) - Reading fresh team state');
      
      const allStatbars = Array.from(document.querySelectorAll('.statbar'));
      
      // SORT BY VERTICAL POSITION
      // Top (smaller Y) = Opponent
      // Bottom (larger Y) = Player
      // This fixes issues where DOM order differs from visual layout
      allStatbars.sort((a, b) => {
          const rectA = a.getBoundingClientRect();
          const rectB = b.getBoundingClientRect();
          return rectA.top - rectB.top;
      });

      console.log('🥋 Found', allStatbars.length, 'statbars');
      
      if (allStatbars.length === 0) {
        // If no statbars and no cache, we can't do anything
        if (!this.cachedTeamState) return null;
        // If we have cache, fallback to it (maybe brief flicker where bars disappeared)
        console.log('🥋 No statbars found, falling back to cache');
      } else {
        const allPokemon = [];
        for (const statbar of allStatbars) {
          const p = this.parseStatbar(statbar);
          if (p) {
            allPokemon.push(p);
          }
        }
        
        // Split by side
        const midpoint = Math.floor(allPokemon.length / 2);
        for (let i = 0; i < allPokemon.length; i++) {
          if (i < midpoint) {
            opponentPokemon.push(allPokemon[i]);
          } else {
            playerPokemon.push(allPokemon[i]);
          }
        }
        
        playerBench = this.readBench();
        opponentBench = Object.values(this.benchHistory); // This is always from log/cache anyway
        
        // UPDATE CACHE
        this.cachedTeamState = {
          playerActive: playerPokemon,
          opponentActive: opponentPokemon,
          playerBench: playerBench,
          // We don't cache opponent bench as it depends on history which is separate
        };
      }
    } 
    
    // If we didn't full parse (or failed to), load from cache
    if ((!shouldFullParse || playerPokemon.length === 0) && this.cachedTeamState) {
        console.log('🥋 Animation Phase - Using CACHED team state');
        playerPokemon = this.cachedTeamState.playerActive;
        opponentPokemon = this.cachedTeamState.opponentActive;
        playerBench = this.cachedTeamState.playerBench;
        opponentBench = Object.values(this.benchHistory);
    }

    // --- ENRICHMENT (Run on both fresh and cached data to ensure HP/Status updates if possible) ---
    // Note: If using cached objects, we might normally miss HP updates during animation. 
    // This is INTENTIONAL to prevent jumping. We only want "settled" states.
    // However, for correct engine logic, we apply the latest history/moves to these objects.

    // Enrich all Pokemon with move history and revealed data
    const allActive = [...playerPokemon, ...opponentPokemon];
    for (const p of allActive) {
      const normalizedSpecies = this.normalizeName(p.species);
      const normalizedName = this.normalizeName(p.name || '');
      
      // Get moves used for either the species or the name
      const usedBySpecies = this.usedMoves[normalizedSpecies] || new Set();
      const usedByName = this.usedMoves[normalizedName] || new Set();
      const mergedMoves = new Set([...usedBySpecies, ...usedByName]);
      p.usedMoves = Array.from(mergedMoves);
    }
    
    // Enrich opponent Pokemon with additional data
    for (const p of opponentPokemon) {
      const speciesKey = p.species.toLowerCase().replace(/[^a-z0-9]/g, '');
      const normalizedName = this.normalizeName(p.name || '');
      
      // Map nickname to species key for better log parsing
      if (normalizedName) {
        this.nicknameMap[normalizedName] = speciesKey;
      }
      
      // Store moves under species key
      if (p.usedMoves && p.usedMoves.length > 0) {
        if (!this.usedMoves[speciesKey]) {
          this.usedMoves[speciesKey] = new Set();
        }
        p.usedMoves.forEach(m => this.usedMoves[speciesKey].add(m));
      }
      
      // Enrich with Log-parsed data
      if (this.revealedData[speciesKey]) {
        if (this.revealedData[speciesKey].teraType) {
          p.teraType = this.revealedData[speciesKey].teraType;
        }
        if (this.revealedData[speciesKey].ability) {
          p.revealedAbility = this.revealedData[speciesKey].ability;
        }
        if (this.revealedData[speciesKey].item) {
          p.revealedItem = this.revealedData[speciesKey].item;
        }
        if (this.revealedData[speciesKey].types) {
           p.types = this.revealedData[speciesKey].types;
        }
      }
      
      // Enrich with stat boosts
      const boostKey = p.name || p.species;
      if (this.boosts && this.boosts[boostKey]) {
        p.boosts = { ...this.boosts[boostKey] };
      } else {
        p.boosts = { atk: 0, def: 0, spa: 0, spd: 0, spe: 0, accuracy: 0, evasion: 0 };
      }
      
      // Get all known moves
      const allKnownMoves = Array.from(new Set([...this.getUsedMoves(speciesKey), ...this.getUsedMoves(normalizedName)]));
      
      // Update bench history
      this.benchHistory[speciesKey] = {
        species: p.species,
        name: p.name,
        lastHp: p.hpPercent,
        lastStatus: p.status,
        usedMoves: allKnownMoves,
        revealedAbility: this.revealedData[speciesKey]?.ability,
        revealedItem: this.revealedData[speciesKey]?.item,
        teraType: this.revealedData[speciesKey]?.teraType,
        displayAbility: this.revealedData[speciesKey]?.ability && typeof SmogonDB !== 'undefined' ? SmogonDB.formatName(this.revealedData[speciesKey].ability) : null,
        displayItem: this.revealedData[speciesKey]?.item && typeof SmogonDB !== 'undefined' ? SmogonDB.formatName(this.revealedData[speciesKey].item) : null
      };
    }

    if (shouldFullParse) {
        // Only try to read controls if we are in the input phase
        this.readControls(playerPokemon, activePromptName);
    } else {
        // If using cache, we might want to carry over available moves from previous turn if same mon?
        // For now, let's leave availableMoves as is (from the cached object)
    }

    // Calculate faint count from bench size and active count
    // In VGC: 4 total Pokemon per side. Faint count = 4 - (active + bench)
    const playerFaints = Math.max(0, 4 - playerPokemon.length - playerBench.length);
    const opponentFaints = Math.max(0, 4 - opponentPokemon.length - Object.keys(this.benchHistory).length);
    const totalFaints = playerFaints + opponentFaints;

    const state = {
      turn: this.readTurn(),
      format: this.readFormat(),
      faintCount: totalFaints,
      lastTurnOrder: this.turnOrder || [],
      player: {
        active: playerPokemon,
        bench: playerBench,
        sideConditions: {}
      },
      opponent: {
        active: opponentPokemon,
        bench: Object.values(this.benchHistory),
        sideConditions: {}
      },
      field: this.readFieldState(), 
      fieldMetadata: this.fieldLogState || {} 
    };
    
    // Apply tooltip data if available (enriches abilities/items)
    this.applyTooltipData(state.player.active, state.opponent.active);

    // === SELF-VERIFICATION BOT ===
    if (controlsVisible) {
        this.verifyState(state, activePromptName);
    }
    
    return state;
  }

  verifyState(state, promptName) {
      if (!state) return;
      const issues = [];
      
      // 1. Critical: Active Pokemon
      if (state.player.active.length === 0) issues.push("⚠️ No active player Pokémon found.");
      
      // 2. Critical: Moves (only if it's our turn)
      const moveCount = state.player.active.reduce((acc, p) => acc + (p.availableMoves ? p.availableMoves.length : 0), 0);
      if (promptName && moveCount === 0) {
          issues.push(`🚨 IT IS YOUR TURN ("${promptName}"), BUT NO MOVES WERE FOUND.`);
      }

      // 3. Side Detection Consistency
      if (promptName) {
           const activeMon = state.player.active.find(p => this.normalizeName(p.name).includes(this.normalizeName(promptName)));
           if (!activeMon) {
               issues.push(`🚨 SIDE MISMATCH: Prompt is for "${promptName}", but that Pokémon is not in your active slot.`);
           }
      }

      // Log Report
      if (issues.length > 0) {
          console.group('🤖 Battle Coach Diagnosis');
          issues.forEach(i => console.warn(i));
          console.log('State Snapshot:', state);
          console.groupEnd();
          
          // Expose to window for user inspection
          window.BATTLE_COACH_LAST_ERROR = { time: new Date().toISOString(), issues, state };
      } else {
          // console.log('✅ State Verification Passed');
      }
  }

  readPlayerActivePokemonPrompt() {
    const prompt = document.querySelector('.whatdo');
    if (prompt) {
      const text = prompt.textContent || '';
      const match = text.match(/What will (.+?) do/i);
      return match ? match[1].trim() : null;
    }
    return null;
  }

  parseBattleLog() {
    const battleLog = document.querySelector('.battle-log .inner, .inner.message-log, .battle-log');
    if (!battleLog) return;

    // innerText is more reliable for line-by-line parsing of Showdown log
    const text = battleLog.innerText || '';
    const lines = text.split('\n');

    // Track Turn Order for Speed Inference
    this.turnOrder = [];
    let lastTurnIndex = -1;
    
    // Find the LAST "Turn X" header to scope to current turn
    for (let i = lines.length - 1; i >= 0; i--) {
        if (lines[i].match(/^Turn \d+/)) {
            lastTurnIndex = i;
            break;
        }
    }
    
    // Calculate which lines belong to the current turn
    const currentTurnLines = (lastTurnIndex !== -1) ? lines.slice(lastTurnIndex) : lines;
    
    // Initialize/Reset Boost Tracking for Replay
    if (!this.boosts) this.boosts = {};
    
    // Initialize Field State Tracking
    this.fieldLogState = {
        weather: null,
        terrain: null,
        trickRoom: null,
        tailwind: { player: null, opponent: null }
    };
    
    // Helper to map log stat names to internal keys
    const statMap = {
        'Attack': 'atk', 'Defense': 'def', 
        'Sp. Atk': 'spa', 'Sp. Def': 'spd', 
        'Speed': 'spe', 'accuracy': 'accuracy', 'evasiveness': 'evasion'
    };
    
    let replayTurn = 0;

    for (const line of lines) {
       // Track Current Turn during Replay
       const turnMatch = line.match(/^Turn (\d+)/);
       if (turnMatch) {
           replayTurn = parseInt(turnMatch[1], 10);
       }
       
       // Speed Tier Inference
       if (lastTurnIndex !== -1 && lines.indexOf(line) > lastTurnIndex) {
           const moveActionMatch = line.match(/(?:The opposing |Foe's |Your |)?(.+?) used /);
           if (moveActionMatch) {
               const moverName = moveActionMatch[1].trim();
               if (!this.turnOrder.includes(moverName)) {
                   this.turnOrder.push(moverName);
               }
           }
       }
       
       if (!line) continue;

       // 6. Stat Change Detection: Replay history to track current stages
       
       // Handle Switch/Faint (Reset Stats)
       const switchResetMatch = line.match(/(?:The opposing |Foe's |Your |Go! )?(.+?) (switched in|sent out|fainted)!/);
       if (switchResetMatch) {
           const monName = switchResetMatch[1].trim();
           // Reset boosts for this pokemon
           this.boosts[monName] = { atk: 0, def: 0, spa: 0, spd: 0, spe: 0, accuracy: 0, evasion: 0 };
       }
       
       // Handle "Stat Rose" (Boost)
       const roseMatch = line.match(/(?:The opposing |Foe's |Your |)?(.+?)'s ([A-Za-z\.\s]+) rose( sharply| drastically)?!/);
       if (roseMatch) {
           const monName = roseMatch[1].trim();
           const statName = roseMatch[2].trim();
           const multiplier = roseMatch[3] ? (roseMatch[3].trim() === 'drastically' ? 3 : 2) : 1;
           const statKey = statMap[statName];
           if (statKey) {
               if (!this.boosts[monName]) this.boosts[monName] = { atk: 0, def: 0, spa: 0, spd: 0, spe: 0, accuracy: 0, evasion: 0 };
               this.boosts[monName][statKey] = Math.min(6, (this.boosts[monName][statKey] || 0) + multiplier);
           }
       }
       
       // Handle "Stat Fell" (Drop)
       const fellMatch = line.match(/(?:The opposing |Foe's |Your |)?(.+?)'s ([A-Za-z\.\s]+) fell( harshly| severely)?!/);
       if (fellMatch) {
           const monName = fellMatch[1].trim();
           const statName = fellMatch[2].trim();
           const multiplier = fellMatch[3] ? (fellMatch[3].trim() === 'severely' ? 3 : 2) : 1;
           const statKey = statMap[statName];
           if (statKey) {
               if (!this.boosts[monName]) this.boosts[monName] = { atk: 0, def: 0, spa: 0, spd: 0, spe: 0, accuracy: 0, evasion: 0 };
               this.boosts[monName][statKey] = Math.max(-6, (this.boosts[monName][statKey] || 0) - multiplier);
           }
       }
       
       // 7. Field Condition Detection
       // Weather
       if (line.includes('The sunlight turned harsh!')) this.fieldLogState.weather = { type: 'sun', startTurn: replayTurn };
       if (line.includes('It started to rain!')) this.fieldLogState.weather = { type: 'rain', startTurn: replayTurn };
       if (line.includes('A sandstorm kicked up!')) this.fieldLogState.weather = { type: 'sand', startTurn: replayTurn };
       if (line.includes('It started to hail!') || line.includes('It started to snow!')) this.fieldLogState.weather = { type: 'snow', startTurn: replayTurn }; // Showdown uses snowGen9 now mostly
       
       if (line.includes('The sunlight faded.') || line.includes('The rain stopped.') || line.includes('The sandstorm subsided.') || line.includes('The hail stopped.') || line.includes('The snow stopped.')) {
           this.fieldLogState.weather = null;
       }
       
       // Terrain
       if (line.includes('Electric Terrain active!')) this.fieldLogState = { ...this.fieldLogState, terrain: { type: 'electric', startTurn: replayTurn } };
       if (line.includes('Grassy Terrain active!')) this.fieldLogState = { ...this.fieldLogState, terrain: { type: 'grassy', startTurn: replayTurn } };
       if (line.includes('Misty Terrain active!')) this.fieldLogState = { ...this.fieldLogState, terrain: { type: 'misty', startTurn: replayTurn } };
       if (line.includes('Psychic Terrain active!')) this.fieldLogState = { ...this.fieldLogState, terrain: { type: 'psychic', startTurn: replayTurn } };
       if (line.includes('The terrain returned to normal.')) this.fieldLogState.terrain = null;
       
       // Trick Room
       if (line.includes('The dimensions became twisted!')) this.fieldLogState.trickRoom = { startTurn: replayTurn };
       if (line.includes('The twisted dimensions returned to normal!')) this.fieldLogState.trickRoom = null;
       
       // Tailwind (Side condition, but useful to track here)
       if (line.includes('blew from behind your team!')) this.fieldLogState.tailwind.player = { startTurn: replayTurn };
       if (line.includes("your team's Tailwind petered out!")) this.fieldLogState.tailwind.player = null;
       
       if (line.includes('blew from behind the opposing team!')) this.fieldLogState.tailwind.opponent = { startTurn: replayTurn };
       if (line.includes("the opposing team's Tailwind petered out!")) this.fieldLogState.tailwind.opponent = null;

      if (!line) continue;
      
      // 1. Ability Detection: "[Prefix] [Pokemon]'s [Ability]"
      // Ex: "The opposing Maushold's Friend Guard!"
      const abilityMatch = line.match(/(?:The opposing |Foe's |Your |)?(.+?)'s ([A-Za-z\s0-9]+)(!)?$/);
      if (abilityMatch) {
         const monName = abilityMatch[1].trim();
         const abilityName = abilityMatch[2].trim();
         
         // Filter out common phrases that look like abilities but aren't
         const ignore = ['attack', 'defense', 'Team', 'Dojo', 'parting shot', 'stat changes', 'speed', 'special attack', 'special defense'];
         if (!abilityName.includes('activated') && !abilityName.includes('restored') && !ignore.includes(abilityName.toLowerCase())) {
             this.recordRevealedData(monName, 'ability', abilityName);
             continue;
         }
      }

      // 2. Item Detection: "[Prefix] [Pokemon]'s [Item] activated!" or "restored its health!"
      const itemMatch = line.match(/(?:The opposing |Foe's |Your |)?(.+?)'s ([A-Za-z\s0-9]+) (activated|restored|healed|ate)/);
      if (itemMatch) {
         const monName = itemMatch[1].trim();
         const itemName = itemMatch[2].trim();
         this.recordRevealedData(monName, 'item', itemName);
         continue;
      }
      
      // 2a. Item Detection: "[Prefix] [Pokemon] restored ... using its [Item]!" (Leftovers/Black Sludge)
      const restoreMatch = line.match(/(?:The opposing |Foe's |Your |)?(.+?) restored .*? using its ([A-Za-z\s0-9]+)!/);
      if (restoreMatch) {
         const monName = restoreMatch[1].trim();
         const itemName = restoreMatch[2].trim();
         this.recordRevealedData(monName, 'item', itemName);
         continue;
      }
      
      // 2b. Item Detection: "[Prefix] [Pokemon] ate its [Item]!"
      const eatMatch = line.match(/(?:The opposing |Foe's |Your |)?(.+?) ate its ([A-Za-z\s0-9]+)!/);
      if (eatMatch) {
         const monName = eatMatch[1].trim();
         const itemName = eatMatch[2].trim();
         this.recordRevealedData(monName, 'item', itemName);
         continue;
      }
      
      // 2c. Item Detection: "hurt by [Pokemon]'s [Item]!" (Rocky Helmet)
      const hurtMatch = line.match(/was hurt by (?:the opposing |foe's |your )?(.+?)'s ([A-Za-z\s0-9]+)!/);
      if (hurtMatch) {
         const monName = hurtMatch[1].trim();
         const itemName = hurtMatch[2].trim();
         this.recordRevealedData(monName, 'item', itemName);
         continue;
      }

      // 2d. Item Detection: "[Pokemon]'s [Item] popped!" (Air Balloon)
      const popMatch = line.match(/(?:The opposing |Foe's |Your |)?(.+?)'s ([A-Za-z\s0-9]+) popped!/);
      if (popMatch) {
         const monName = popMatch[1].trim();
         const itemName = popMatch[2].trim();
         this.recordRevealedData(monName, 'item', itemName);
         continue;
      }

      // Broaden regex to capture "The opposing", "Foe's", "Your", or just the name
      // Use a more aggressive match for the Pokemon name part to catch nicknames/species with special chars
      const moveMatch = line.match(/(?:The opposing |Foe's |Your |)?(.+?) used ([A-Za-z0-9\s\-',]+)!/);
      if (moveMatch) {
        const pokemonDisplayName = moveMatch[1].trim();
        const moveName = moveMatch[2].trim();
        const normalized = this.normalizeName(pokemonDisplayName);
        
        if (moveName && moveName !== 'Recharge' && moveName !== 'Struggle') {
          // 1. Initial attribution to the name/nickname found in log
          if (!this.usedMoves[normalized]) { // Renamed from revealedMoves
            this.usedMoves[normalized] = new Set(); // Renamed from revealedMoves
          }
          this.usedMoves[normalized].add(moveName); // Renamed from revealedMoves
          
          // 2. Cross-attribute to species if we have a nickname mapping
          const speciesKey = this.nicknameMap[normalized];
          if (speciesKey && speciesKey !== normalized) {
            if (!this.usedMoves[speciesKey]) { // Renamed from revealedMoves
              this.usedMoves[speciesKey] = new Set(); // Renamed from revealedMoves
            }
            this.usedMoves[speciesKey].add(moveName); // Renamed from revealedMoves
          }
          
          // 3. Update benchHistory immediately
          const targetKey = speciesKey || normalized;

          if (this.benchHistory[targetKey]) {
            const history = this.benchHistory[targetKey];
            if (!history.usedMoves) history.usedMoves = [];
            if (!history.usedMoves.includes(moveName)) {
              history.usedMoves.push(moveName);
            }
          }
        }
      }

      // 4. Switch Detection: "[Prefix] [Pokemon] switched in!" or "sent out [Pokemon]!"
      const switchMatch = line.match(/(?:The opposing |Foe's |Your |Go! )?(.+?) (switched in|sent out)!/);
      if (switchMatch) {
          const monName = switchMatch[1].trim();
          const normalized = this.normalizeName(monName);
          
          const speciesKey = this.nicknameMap[normalized] || normalized;
          if (!this.benchHistory[speciesKey]) {
              this.benchHistory[speciesKey] = {
                  species: monName, // Best guess if not in history
                  name: monName,
                  lastHp: 100,
                  usedMoves: []
              };
          }
      }

      // 5. Tera Type Detection: "[Pokemon] Terastallized into the [Type] type!"
      // Pattern: "Khabib Terastallized into the Ghost type!"
      const teraMatch = line.match(/(?:The opposing |Foe's |Your )?(.+?) Terastallized into the ([a-zA-Z]+) type!/);
      if (teraMatch) {
          const monName = teraMatch[1].trim();
          const teraType = teraMatch[2].trim();
          this.recordRevealedData(monName, 'tera', teraType);
      }
    }
  }

  recordRevealedData(name, type, value) {
     const normalized = this.normalizeName(name);
     // nicknameMap uses normalized nicknames to point to species keys
     const speciesKey = this.nicknameMap[normalized] || normalized;
     
     if (!this.revealedData[speciesKey]) {
         this.revealedData[speciesKey] = {};
     }
     
     // Only overwrite if new or better? For now just take latest
     if (type === 'ability') {
         // Filter out 'Protosynthesis' msg if we already have a specific one? No, usually it's fine.
         this.revealedData[speciesKey].ability = value;
     } else if (type === 'item') {
         this.revealedData[speciesKey].item = value;
     } else if (type === 'tera') {
         this.revealedData[speciesKey].teraType = value;
     }
     
     // Also update bench history live if possible
     if (this.benchHistory[speciesKey]) {
         if (type === 'ability') this.benchHistory[speciesKey].revealedAbility = value;
         if (type === 'item') this.benchHistory[speciesKey].revealedItem = value;
         if (type === 'tera') this.benchHistory[speciesKey].teraType = value;
     }
  }

  normalizeName(name) {
    if (!name) return '';
    let normalized = name
      .replace(/^The opposing /i, '')
      .replace(/^Foe's /i, '')
      .replace(/^Your /i, '')
      .replace(/^Tera-[A-Z][a-z]+(\s+|$)/i, '') 
      .replace(/\s*L\d+$/i, '')
      .replace(/[♀♂\*]/g, '')
      .trim()
      .toLowerCase()
      .replace(/[^a-z0-9]/g, ''); // Strips EVERYTHING except letters and numbers for unified lookup

    // VGC Competitive Mapping: Redirection for species whose base name is almost always a regional form in VGC
    const VGC_FORM_MAP = {
      'ninetales': 'ninetalesalola',
      'arcanine': 'arcaninehisui',
      'raichu': 'raichualola',
      'persian': 'persianalola',
      'muk': 'mukalola',
      'marowak': 'marowakalola'
    };

    return VGC_FORM_MAP[normalized] || normalized;
  }

  getUsedMoves(normalizedName) {
    return this.usedMoves[normalizedName] || new Set();
  }

  /**
   * Read available moves from the control buttons
   */
  readControls(playerPokemon, activePromptName) {
    const controls = document.querySelector('.battle-controls');
    if (!controls) return;

    // 1. Move Buttons
    const moveButtons = controls.querySelectorAll('button[name="chooseMove"]');
    if (moveButtons.length > 0) {
        const moves = [];
        moveButtons.forEach(btn => {
            // Try data attribute first (most reliable)
            let moveName = btn.getAttribute('data-move');
            let type = null;
            
            // Fallback to text parsing
            if (!moveName) {
                // Button text usually: "MoveName Type PP/PP" or similar
                // Clone and remove child elements to get raw text might be hard
                // Just splitting by newline is safer
                const text = btn.innerText; 
                const lines = text.split('\n');
                moveName = lines[0].trim();
                if (lines[1]) type = lines[1].trim();
            } else {
                // Try to get type from class (e.g. "type-Fire")
                const typeClass = Array.from(btn.classList).find(c => c.startsWith('type-'));
                if (typeClass) type = typeClass.replace('type-', '');
            }

            if (moveName && moveName !== 'Struggle' && moveName !== 'Recharge') {
                moves.push({ name: moveName, type: type });
            }
        });

        // Assign to the correct Pokemon
        if (moves.length > 0) {
            let targetMon = null;
            
            // Try to match by prompt name (e.g. "What will Arcanine do?")
            if (activePromptName) {
                const promptLower = activePromptName.toLowerCase();
                targetMon = playerPokemon.find(p => {
                    return (p.name || '').toLowerCase().includes(promptLower) || 
                           (p.species || '').toLowerCase().includes(promptLower);
                });
            }
            
            // Fallback: If single active pokemon, assign to it
            // (Most common case for singles)
            if (!targetMon && playerPokemon.length === 1) {
                targetMon = playerPokemon[0];
            }
            
            // If found, update its available moves
            if (targetMon) {
                targetMon.availableMoves = moves;
                
                // Also update our persistent cache so we remember them next turn if buttons act weird
                const speciesKey = this.nicknameMap[this.normalizeName(targetMon.name)] || this.normalizeName(targetMon.species);
                if (speciesKey) {
                   if (!this.benchHistory[speciesKey]) {
                       this.benchHistory[speciesKey] = { usedMoves: [] };
                   }
                   // Merge valid moves
                   moves.forEach(m => {
                       if (!this.benchHistory[speciesKey].usedMoves) this.benchHistory[speciesKey].usedMoves = [];
                       if (!this.benchHistory[speciesKey].usedMoves.includes(m.name)) {
                           this.benchHistory[speciesKey].usedMoves.push(m.name);
                       }
                   });
                }
            }
        }
    }
  }



  /**
   * Read bench Pokemon
   */
  readBench() {
    const bench = [];
    // Scope to the controls to avoid opponent switch buttons in spectator mode
    const controls = document.querySelector('.controls, .movemenu, .switchmenu');
    const switchButtons = controls ? controls.querySelectorAll('button[name="chooseSwitch"]') : [];
    
    for (const button of switchButtons) {
      const name = button.textContent?.trim()?.split('\n')[0]?.replace('Switch to ', '');
      if (name) {
        // Skip buttons that are visually fainted or disabled
        const isFainted = button.disabled || button.className.includes('disabled') || button.textContent.includes('(fainted)');
        if (!isFainted) {
            bench.push({ species: name, fainted: false });
        }
      }
    }
    return bench;
  }

  parseStatbar(statbar) {
    if (!statbar) return null;

    // DEBUG: Log all potential sources of Pokemon identity
    console.log('[Battle Coach] Statbar debug:', {
        title: statbar.getAttribute('title'),
        strongText: statbar.querySelector('strong')?.textContent,
        strongTitle: statbar.querySelector('strong')?.getAttribute('title'),
        classList: Array.from(statbar.classList).join(' ')
    });

    const nameElement = statbar.querySelector('strong');
    
    // Name Corruption Fix: Statbars often inject <span class="teratype">Tera-Ghost</span>
    // inside the <strong> tag. We iterate over child nodes to collect ONLY the text nodes,
    // explicitly skipping any <span>, <i>, or <img> elements.
    let rawName = 'Unknown';
    if (nameElement) {
        let textParts = [];
        for (const node of nameElement.childNodes) {
          // nodeType 3 is Text
          if (node.nodeType === 3) {
            textParts.push(node.textContent);
          }
        }
        rawName = textParts.join('').trim();
        
        // If the above found nothing (e.g. name is somehow inside a span), fallback to textContent
        if (!rawName) {
          rawName = nameElement.textContent.trim().split(' ')[0];
        }
    }
    
    // Aggressive Cleaning: Strip Showdown prefixes and suffixes immediately
    // But FIRST - detect ownership!
    let ownership = 'unknown'; // 'player', 'opponent', 'unknown'
    if (rawName.match(/^your /i)) ownership = 'player';
    else if (rawName.match(/^(the opposing |foe's )/i)) ownership = 'opponent';

    // Check 2: Title/Tooltip attributes (More reliable when text is truncated or styled)
    const titleAttr = statbar.getAttribute('title') || statbar.querySelector('strong')?.getAttribute('title') || '';
    if (titleAttr) {
        if (titleAttr.match(/^your /i)) ownership = 'player';
        else if (titleAttr.match(/^(the opposing |foe's )/i)) ownership = 'opponent';
    }

    let cleanName = rawName
      .replace(/^The opposing /i, '')
      .replace(/^Foe's /i, '')
      .replace(/^Your /i, '')
      .replace(/^Tera-[A-Z][a-z]+(\s+|$)/i, '') // Catch "Tera-Flying" if it was part of the text node
      .replace(/\s*L\d+$/i, '')
      .replace(/[♀♂\*]/g, '')
      .trim();

    let species = cleanName;

    // Nickname Handling: Find the sprite image and extract species from alt text
    // Strategy 1: Check the statbar's parent container for sprites
    const pokemonContainer = statbar.closest('.pokemon, .pob-pokemon, .innerbattle > div') || 
                             statbar.parentElement;
    
    // Try multiple selectors to find the sprite image
    let spriteImg = null;
    
    if (pokemonContainer) {
      const imgs = Array.from(pokemonContainer.querySelectorAll('img'));
      spriteImg = imgs.find(img => {
          const alt = img.getAttribute('alt');
          const src = img.getAttribute('src') || '';
          // Look for Pokemon sprites by src pattern - not gender/type icons
          return alt && !['M', 'F', '♂', '♀', '(', ')'].includes(alt) && 
                 !src.includes('gender') && !src.includes('type') &&
                 (src.includes('sprites') || src.includes('pokemon') || src.includes('ani'));
      });
    }
    
    // Strategy 2: Search the entire battle field for images matching this statbar's side
    if (!spriteImg) {
      const battleContainer = document.querySelector('.innerbattle');
      if (battleContainer) {
        const sideClass = statbar.classList.contains('lstatbar') ? 'leftbar' : 'rightbar';
        // Find all sprites on the same side
        const spriteContainers = battleContainer.querySelectorAll(`.${sideClass}, .pokemon`);
        for (const container of spriteContainers) {
          const imgs = container.querySelectorAll('img');
          for (const img of imgs) {
            const alt = img.getAttribute('alt');
            const src = img.getAttribute('src') || '';
            if (alt && alt.length > 1 && 
                (src.includes('sprites') || src.includes('pokemon') || src.includes('ani')) &&
                !src.includes('gender') && !src.includes('type')) {
              spriteImg = img;
              break;
            }
          }
          if (spriteImg) break;
        }
      }
    }
    
    if (spriteImg) {
      const alt = spriteImg.getAttribute('alt');
      const src = spriteImg.getAttribute('src') || '';
      
      // Strategy A: Parse species from sprite src URL (most reliable for forms)
      // Showdown URLs: /sprites/gen5ani/ursalunabloodmoon.gif or /sprites/ani/ursaluna-bloodmoon.gif
      const srcMatch = src.match(/\/([a-z0-9\-]+)(?:\.png|\.gif)/i);
      if (srcMatch) {
        let srcSpecies = srcMatch[1];
        // Convert to display format: "ursalunabloodmoon" -> "Ursaluna-Bloodmoon"
        // Common form suffixes
        srcSpecies = srcSpecies
          .replace(/bloodmoon/i, '-Bloodmoon')
          .replace(/alola/i, '-Alola')
          .replace(/galar/i, '-Galar')
          .replace(/hisui/i, '-Hisui')
          .replace(/paldea/i, '-Paldea')
          .replace(/mega/i, '-Mega')
          .replace(/gmax/i, '-Gmax');
        // Capitalize first letter
        species = srcSpecies.charAt(0).toUpperCase() + srcSpecies.slice(1);
      }
      
      // Strategy B: Parse species from alt text (fallback)
      if (alt && (!species || species.length < 2)) {
        // Alt text is usually "PokemonName" or "Nickname (PokemonName)"
        const speciesMatch = alt.match(/\(([^)]+)\)/);
        if (speciesMatch) {
          species = speciesMatch[1];
        } else {
          // Alt text might just be the species name directly
          species = alt;
        }
      }
    }

    // Clean any remaining visual prefixes from species name
    species = species.replace(/^(The opposing |Foe's |Your |Tera-[A-Z][a-z]+(\s+|$))/i, '').trim();
    
    // Final check: If species still has "Tera-" in it, strip it one last time
    species = species.replace(/^Tera-[A-Z][a-z]+(\s+|$)/i, '').trim();
    cleanName = cleanName.replace(/^Tera-[A-Z][a-z]+(\s+|$)/i, '').trim();
    
    // Final fallback for species: title text on statbar
    if (!species || species.length < 2 || species === cleanName) {
      if (statbar.title && statbar.title.includes('(')) {
         const match = statbar.title.match(/\(([^)]+)\)/);
         if (match) species = match[1];
      }
    }
    
    // Last resort: Check the tooltip/title attribute for the Pokemon name
    if (!species || species.length < 2 || species === cleanName) {
      const titleAttr = statbar.getAttribute('title') || statbar.querySelector('strong')?.getAttribute('title');
      if (titleAttr) {
        // Title often contains "Nickname (Species)" or just "Species"
        const match = titleAttr.match(/\(([^)]+)\)/);
        if (match) {
          species = match[1];
        } else if (titleAttr.length > 2 && titleAttr !== cleanName) {
          species = titleAttr.split(' ')[0]; // Take first word as species guess
        }
      }
    }

    let hpPercent = 100;
    const hpBar = statbar.querySelector('.hpbar .hp');
    if (hpBar && hpBar.style.width) {
      const width = parseFloat(hpBar.style.width);
      if (!isNaN(width)) hpPercent = Math.min(100, Math.max(0, Math.round(width)));
    }
    
    const hpText = statbar.querySelector('.hptext');
    if (hpText) {
      const pctMatch = hpText.textContent.match(/(\d+)%/);
      if (pctMatch) hpPercent = Math.min(100, parseInt(pctMatch[1]));
    }

    const statusElement = statbar.querySelector('.status');
    let status = null;
    if (statusElement) {
      const statusClass = statusElement.className.split(' ').find(c => ['brn', 'par', 'slp', 'frz', 'psn', 'tox'].includes(c));
      status = statusClass || null;
    }
    
    // Parse stat boosts
    const boosts = {};
    const boostElements = statbar.querySelectorAll('.status span, .boost');
    boostElements.forEach(el => {
      const text = el.textContent.trim();
      const match = text.match(/([+-]\d+)\s*(Atk|Def|SpA|SpD|Spe|Acc|Eva)/i);
      if (match) {
        const val = parseInt(match[1]);
        const stat = match[2].toLowerCase();
        boosts[stat] = (boosts[stat] || 0) + val;
      }
    });

    const teraElement = statbar.querySelector('.teratype');

    return {
      species,
      name: cleanName, // The nickname or raw display name
      hpPercent,
      status,
      boosts,
      terastallized: !!teraElement,
      teraType: teraElement?.title?.replace('Tera Type: ', '') || null,
      usedMoves: [],
      availableMoves: [],
      ownership, // NEW field
      revealedAbility: this.revealedData[this.normalizeName(species)]?.ability || null,
      revealedItem: this.revealedData[this.normalizeName(species)]?.item || null,
      displayAbility: this.revealedData[this.normalizeName(species)]?.ability && typeof SmogonDB !== 'undefined' ? SmogonDB.formatName(this.revealedData[this.normalizeName(species)].ability) : null,
      displayItem: this.revealedData[this.normalizeName(species)]?.item && typeof SmogonDB !== 'undefined' ? SmogonDB.formatName(this.revealedData[this.normalizeName(species)].item) : null
    };
  }

  readTurn() {
    const turnEl = document.querySelector('.turn');
    if (turnEl) {
      const match = turnEl.textContent.match(/Turn\s*(\d+)/i);
      if (match) return parseInt(match[1]);
    }
    return 1;
  }

  /**
   * Determine format based on number of active statbars
   */
  readFormat() {
    const statbars = document.querySelectorAll('.statbar');
    return statbars.length <= 2 ? 'SINGLES' : 'DOUBLES';
  }

  /**
   * Read field conditions like weather and terrain
   */
  readFieldState() {
    const field = { weather: null, terrain: null, trickRoom: false, tailwind: false };
    const weatherEl = document.querySelector('.weather');
    if (weatherEl) {
      const text = weatherEl.textContent?.toLowerCase() || '';
      if (text.includes('sun') || text.includes('harsh sunshine')) field.weather = 'sun';
      else if (text.includes('rain')) field.weather = 'rain';
      else if (text.includes('sand')) field.weather = 'sand';
      else if (text.includes('snow') || text.includes('hail')) field.weather = 'snow';
    }
    const terrainEl = document.querySelector('.terrain');
    if (terrainEl) {
       const text = terrainEl.textContent?.toLowerCase() || '';
       if (text.includes('electric')) field.terrain = 'electric';
       else if (text.includes('grassy')) field.terrain = 'grassy';
       else if (text.includes('misty')) field.terrain = 'misty';
       else if (text.includes('psychic')) field.terrain = 'psychic';
    }

    // Modern Showdown Themes often put Trick Room/Tailwind in these containers
    const battleStatus = document.querySelector('.battle-status, .room-status, .innerbattle');
    if (battleStatus) {
        const text = battleStatus.textContent?.toLowerCase() || '';
        if (text.includes('trick room')) field.trickRoom = true;
        // Tailwind is often shown in the statbars or side condition areas
        if (text.includes('tailwind')) field.tailwind = true;
    }

    // Check for specific icons or spans that Showdown uses for conditions
    const statusSpans = document.querySelectorAll('.statbar-status, .side-condition, .weather-status');
    statusSpans.forEach(span => {
        const text = span.textContent?.toLowerCase() || '';
        if (text.includes('trick room')) field.trickRoom = true;
        if (text.includes('tailwind')) field.tailwind = true;
    });

    return field;
  }

  applyTooltipData(playerActive, opponentActive) {
    const tooltip = document.querySelector('.tooltip');
    if (!tooltip || tooltip.style.display === 'none') return;

    const text = tooltip.innerText;
    if (!text) return;

    // Extract Pokemon name from tooltip title (usually first line, bold)
    const titleMatch = text.match(/^([A-Za-z0-9\s\-\(\)\.]+)(?:\s+\(Level \d+\))?/);
    if (!titleMatch) return;
    
    let monName = titleMatch[1].trim();
    // Handle "Hypno (Foe)" -> "Hypno"
    monName = monName.replace(/ \((Foe|Opposing|Your)\)$/, '').trim();
    const normalizedMon = this.normalizeName(monName);

    // Find matching Pokemon in active sets
    const match = [...playerActive, ...opponentActive].find(p => 
      this.normalizeName(p.name) === normalizedMon || 
      this.normalizeName(p.species) === normalizedMon
    );

    if (match) {
      // Parse Types from Icons (IMG with alt="Type")
      // Tooltips often show types as images
      const typeImgs = Array.from(tooltip.querySelectorAll('img[alt]'));
      const foundTypes = typeImgs
        .map(img => img.alt.toLowerCase())
        .filter(t => ['normal', 'fire', 'water', 'electric', 'grass', 'ice', 'fighting', 'poison', 'ground', 'flying', 'psychic', 'bug', 'rock', 'ghost', 'dragon', 'dark', 'steel', 'fairy'].includes(t));
      
      if (foundTypes.length > 0) {
        // Persist type data since database might be outdated
        this.recordRevealedData(match.name, 'types', foundTypes);
        match.types = foundTypes;
        if (match.species && match.species !== match.name) {
           this.recordRevealedData(match.species, 'types', foundTypes);
        }
      }

      // Parse Possible Abilities
      if (text.includes('Possible abilities:')) {
        const abilitiesStr = text.split('Possible abilities:')[1].split('\n')[0];
        match.possibleAbilities = abilitiesStr.split(',').map(a => a.trim());
      } else if (text.includes('Ability:')) {
        const abilityStr = text.split('Ability:')[1].split('\n')[0].trim();
        // Since it's confirmed in tooltip, treat it as revealed!
        this.recordRevealedData(match.name, 'ability', abilityStr);
        // Also populate possibleAbilities for consistency in viewing
        match.possibleAbilities = [abilityStr];
        match.revealedAbility = abilityStr;
      }

      // Parse Item
      if (text.includes('Item:')) {
        const itemStr = text.split('Item:')[1].split('\n')[0].trim();
        if (itemStr !== 'None') {
          this.recordRevealedData(match.name, 'item', itemStr);
          match.possibleItems = [itemStr];
          match.revealedItem = itemStr;
        } else {
             match.possibleItems = []; 
        }
      }
    }
  }
}

if (typeof module !== 'undefined') {
  module.exports = ShowdownStateReader;
}
