/**
 * Battle Coach - Content Script
 * READ-ONLY: Only observes the page and sends battle state to background
 * Does NOT modify the DOM in any way
 */

class DojoCompanion {
  constructor() {
    this.stateReader = new ShowdownStateReader();
    this.lastStateJson = null;
    this.hasFoundBattle = false;
  }

  init() {
    console.log('🥋 Battle Coach initialized');
    chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
      if (message.type === 'requestState') {
        const state = this.stateReader.readState();
        if (state) {
          this.hasFoundBattle = true;
          this.lastStateJson = JSON.stringify(state);
        }
        // Always respond, even if state is null
        sendResponse({ state: state || null });
        return true; // Keep channel open for async
      }
      
      if (message.type === 'forceRefresh') {
        console.log('🥋 Force refresh - resetting entire battle state...');
        
        // Reset ALL internal state reader caches
        this.stateReader.usedMoves = {};
        this.stateReader.benchHistory = {};
        this.stateReader.nicknameMap = {};
        this.stateReader.revealedData = {};
        this.stateReader.boosts = {};
        this.stateReader.lastTurn = 0;
        this.stateReader.turnOrder = [];
        this.stateReader.fieldLogState = null;
        this.lastStateJson = null;
        this.hasFoundBattle = false;
        
        console.log('🥋 All caches cleared, re-reading battle state from DOM...');
        
        // Re-read state from fresh DOM
        const freshState = this.stateReader.readState();
        if (freshState) {
          this.hasFoundBattle = true;
          this.lastStateJson = JSON.stringify(freshState);
          console.log('🥋 Fresh state read:', freshState.player?.active?.length, 'player,', freshState.opponent?.active?.length, 'opponent');
        } else {
          console.log('🥋 No battle state found after refresh');
        }
        sendResponse({ state: freshState || null });
        return true;
      }
    });
    this.startPolling();
  }

  startPolling() {
    // Check every 2 seconds
    setInterval(() => {
      this.checkBattleState();
    }, 2000);
    
    // Check more frequently at start
    setTimeout(() => this.checkBattleState(), 500);
    setTimeout(() => this.checkBattleState(), 1500);
    setTimeout(() => this.checkBattleState(), 3000);
  }

  checkBattleState() {
    const state = this.stateReader.readState();
    
    // BATTLE END DETECTION:
    // If we get NULL and previously had a battle, count consecutive nulls.
    // After 3 consecutive nulls (6 seconds), assume battle ended and reset.
    if (!state) {
      if (this.hasFoundBattle) {
        this.nullStateCount = (this.nullStateCount || 0) + 1;
        
        // After 3 consecutive null states (~6 seconds), assume battle ended
        if (this.nullStateCount >= 3) {
          console.log('🥋 Battle ended, resetting to waiting state');
          this.hasFoundBattle = false;
          this.lastStateJson = null;
          this.nullStateCount = 0;
          this.sendStateToBackground(null); // Tell panel to show waiting
        }
        return;
      }
      return; 
    }
    
    // Valid state found, reset null counter
    this.nullStateCount = 0;
    this.hasFoundBattle = true;
    const stateJson = JSON.stringify(state);
    
    if (stateJson !== this.lastStateJson) {
      console.log('🥋 State changed, sending to background...');
      this.lastStateJson = stateJson;
      this.sendStateToBackground(state);
    }
  }

  sendStateToBackground(state) {
    try {
      chrome.runtime.sendMessage(
        { type: 'battleState', state: state },
        (response) => {
          if (chrome.runtime.lastError) {
            console.log('🥋 Send error:', chrome.runtime.lastError.message);
          } else {
            console.log('🥋 State sent successfully');
          }
        }
      );
    } catch (e) {
      console.log('🥋 Exception sending state:', e);
    }
  }
}

// Initialize
if (document.readyState === 'loading') {
  document.addEventListener('DOMContentLoaded', () => {
    const companion = new DojoCompanion();
    companion.init();
  });
} else {
  const companion = new DojoCompanion();
  companion.init();
}

