/**
 * Battle Coach - Service Worker
 * Handles messaging between content script and side panel
 */

// Store the latest battle state
let currentBattleState = null;
let connectedPorts = [];

// Track open state for toggling
let openSidePanels = new Set();

// Listen for connections from the side panel to track open state
chrome.runtime.onConnect.addListener((port) => {
  if (port.name === 'sidepanel') {
    const tabId = port.sender?.tab?.id;
    if (tabId) openSidePanels.add(tabId);
    
    console.log('🥋 Side panel connected for tab:', tabId);
    connectedPorts.push(port);
    
    if (currentBattleState) {
      port.postMessage({ type: 'stateUpdate', state: currentBattleState });
    }
    
    port.onDisconnect.addListener(() => {
      connectedPorts = connectedPorts.filter(p => p !== port);
      if (tabId) openSidePanels.delete(tabId);
      console.log('🥋 Side panel disconnected for tab:', tabId);
    });
  }
});

// Listen for messages from content script
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  if (message.type === 'battleState') {
    currentBattleState = message.state;
    
    // Forward to all connected side panels
    connectedPorts.forEach(port => {
      try {
        port.postMessage({ type: 'stateUpdate', state: currentBattleState });
      } catch (e) {
        console.log('🥋 Failed to send to port:', e);
      }
    });
    
    sendResponse({ received: true });
  }
  
  if (message.type === 'getState') {
    sendResponse({ state: currentBattleState });
  }
  
  return true; // Keep channel open for async response
});

// Toggle side panel when clicking the extension icon
chrome.action.onClicked.addListener(async (tab) => {
  if (openSidePanels.has(tab.id)) {
    console.log('🥋 Panel already open, closing...');
    // Workaround to "close" side panel: disable and then re-enable
    await chrome.sidePanel.setOptions({ tabId: tab.id, enabled: false });
    await chrome.sidePanel.setOptions({ tabId: tab.id, enabled: true });
    openSidePanels.delete(tab.id);
  } else {
    console.log('🥋 Opening side panel...');
    try {
      await chrome.sidePanel.open({ tabId: tab.id });
    } catch (e) {
      console.log('🥋 Error opening side panel:', e);
    }
  }
});


// Enable side panel for Showdown tabs
chrome.tabs.onUpdated.addListener((tabId, changeInfo, tab) => {
  if (tab.url?.includes('play.pokemonshowdown.com')) {
    chrome.sidePanel.setOptions({
      tabId,
      path: 'src/sidepanel/panel.html',
      enabled: true
    });
  }
});
